import axios from "axios";
import { getAccessToken } from "./auth.js";
import fs from "fs";
import dotenv from "dotenv";

dotenv.config();

const ADOBE_SIGN_API_URL = process.env.ADOBE_SIGN_API_URL;

export const createAgreement = async (agreementInfo) => {
  const accessToken = await getAccessToken();
  try {
    const response = await axios.post(
      `${ADOBE_SIGN_API_URL}/agreements`,
      agreementInfo,
      {
        headers: { Authorization: `Bearer ${accessToken}` },
      }
    );
    return response.data;
  } catch (error) {
    console.error(
      "Error creating agreement:",
      error.response?.data || error.message
    );
    throw error;
  }
};

export const getAgreementStatus = async (agreementId) => {
  const accessToken = await getAccessToken();
  try {
    const response = await axios.get(
      `${ADOBE_SIGN_API_URL}/agreements/${agreementId}`,
      {
        headers: { Authorization: `Bearer ${accessToken}` },
      }
    );
    return response.data;
  } catch (error) {
    console.error(
      "Error getting agreement status:",
      error.response?.data || error.message
    );
    throw error;
  }
};

export const downloadAgreement = async (agreementId) => {
  const accessToken = await getAccessToken();
  try {
    const response = await axios.get(
      `${ADOBE_SIGN_API_URL}/agreements/${agreementId}/combinedDocument`,
      {
        headers: { Authorization: `Bearer ${accessToken}` },
        responseType: "arraybuffer",
      }
    );

    fs.writeFileSync(`signed_document_${agreementId}.pdf`, response.data);
    console.log(`Signed document saved as signed_document_${agreementId}.pdf`);
  } catch (error) {
    console.error(
      "Error downloading agreement:",
      error.response?.data || error.message
    );
    throw error;
  }
};

const uploadLibraryDocument = async (filePath, fileName) => {
  const accessToken = await getAccessToken();
  const fileBuffer = fs.readFileSync(filePath);

  try {
    const response = await axios.post(
      `${ADOBE_SIGN_API_URL}/libraryDocuments`,
      {
        fileInfos: [
          {
            transientDocumentId: await uploadTransientDocument(
              filePath,
              fileName
            ),
          },
        ],
        name: fileName,
        sharingMode: "ACCOUNT",
      },
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          "Content-Type": "application/json",
        },
      }
    );

    console.log("Library Document ID:", response.data.id);
    return response.data.id;
  } catch (error) {
    console.error(
      "Error uploading document to library:",
      error.response?.data || error.message
    );
    throw error;
  }
};

const uploadTransientDocument = async (filePath, fileName) => {
  const accessToken = await getAccessToken();
  const fileBuffer = fs.readFileSync(filePath);

  const response = await axios.post(
    `${ADOBE_SIGN_API_URL}/transientDocuments`,
    fileBuffer,
    {
      headers: {
        Authorization: `Bearer ${accessToken}`,
        "Content-Type": "application/pdf",
        "Content-Disposition": `attachment; filename="${fileName}"`,
      },
    }
  );

  console.log("Transient Document ID:", response.data.transientDocumentId);
  return response.data.transientDocumentId;
};
