import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  whereCondition,
  countQueryCondition,
  deleteSettingRecord,
  decodeSingle_statement,
  getFirstCreatedAndAllOrganizationIds,
  encodeAndStringifyFields,
  uniqueIdGenerator,
  decodeAndParseFields,
  getFilterConditions,
  getListingData,
  updateQueryBuilder,
  createQueryBuilder,
} from "../helper/general.js";
import Category from "../sequelize/CategorySchema.js";

/** Function to create or update Category */
export const createUpdateCategory = async (req, res) => {
  const { id, organization } = req.body;

  if (id && organization.length === 0) {
    return sendResponse(res, 404, "Please Select at least one business Structure");
  }

  if (!id) {
    const result = await getFirstCreatedAndAllOrganizationIds();

    if (!result.firstOrgId) {
      return sendResponse(res, 404, "No organization found.");
    }
    const unique_id = await uniqueIdGenerator(
      result.firstOrgId,
      req.body?.department || null,
      "Category",
      "category",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.organization = organization.length > 0 ? organization : result.organizationIds;
  }

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id ? updateQueryBuilder(Category, req.body) : createQueryBuilder(Category, req.body);
  const [result] = await db.query(query, values);
  await insertActivityLog(req.user.sessionid, status, "Category", id ? id : result.insertId);
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

/**Function to view all and single Category */
export const viewCategory = async (req, res) => {
  let { organizationCondition, condition } = await getFilterConditions(req, db, "category");

  const searchTableName = [
    "category.name",
    "category.description",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = category.created_by",
    },
    {
      type: "left",
      targetTable: "category AS parent_category",
      onCondition: "parent_category.id = category.parent",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const CategoryQuery = `SELECT category.unique_id, category.id, category.created_by as created_by_id, category.name,parent_category.name AS parent_name , category.description, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, category.organization as organizations, category.parent FROM category ${joinsRecord} WHERE category.deleted = 0 ${searchCondition} ${organizationCondition} ${condition}`;
  let [Category] = await db.query(CategoryQuery);

  Category = await getListingData(Category);

  // async function fetchSubChildren(parentId) {
  //   const [subChildren] = await db.query(`SELECT * FROM category WHERE deleted = 0 AND parent = ?`, [parentId]);

  //   // Only populate sub-children if any are found, and no need to recurse if not needed
  //   for (const child of subChildren) {
  //     child.description = await decodeSingle_statement(child.description);
  //     child.sub_children = await fetchSubChildren(child.id);
  //   }
  //   return subChildren;
  // }

  // // Map through Category and only populate the parent along with its sub-children.
  // const nestedCategory = [];

  // for (const item of Category) {
  //   item.description = await decodeSingle_statement(item.description);
  //   if (item.parent === null) {
  //     item.sub_children = await fetchSubChildren(item.id);
  //     nestedCategory.push(item);
  //   }
  // }

  const totalRecord = await countQueryCondition(CategoryQuery);
  return sendResponse(res, 200, Category, totalRecord);
};

export const viewCategoryDropDown = async (req, res) => {
  try {
    // parent records
    let fetchParentQuery = `
      SELECT id, name, parent, organization 
      FROM category 
      WHERE deleted = 0 AND parent IS NULL
    `;
    let [parentRecords] = await db.query(fetchParentQuery);

    if (parentRecords.length === 0) {
      return sendResponse(res, 200, []);
    }

    const fetchChildrenRecursively = async (parentId) => {
      let fetchChildQuery = `
        SELECT id, name, parent, organization 
        FROM category 
        WHERE deleted = 0 AND parent = ?
      `;
      const [children] = await db.query(fetchChildQuery, [parentId]);

      for (const child of children) {
        const grandchildren = await fetchChildrenRecursively(child.id);
        children.push(...grandchildren);
      }
      return children;
    };

    for (const parent of parentRecords) {
      const children = await fetchChildrenRecursively(parent.id);
      parentRecords.push(...children);
    }

    parentRecords = await decodeAndParseFields(parentRecords);

    return sendResponse(res, 200, parentRecords);
  } catch (error) {
    return sendResponse(res, 500, { error: "Internal Server Error" });
  }
};

export const getParentCategoryList = async (req, res) => {
  let { filter } = req.query;
  filter = typeof filter === "string" ? JSON.parse(filter) : filter;
  let { organizations } = filter;
  if (!organizations || organizations.length == 0) {
    const [arrOrg] = await db.query(`SELECT id FROM organization WHERE deleted = 0`);
    organizations = arrOrg.map((item) => item.id);
  }
  organizations = typeof organizations == "string" ? JSON.parse(organizations) : organizations;

  const categoryQuery = `
    SELECT *
    FROM category
    WHERE 
        ${organizations.map(() => `JSON_CONTAINS(organization, CAST(? AS JSON), '$')`).join(" AND ")}
        AND deleted = 0;
`;

  const params = [...organizations];
  let [parentCategoryList] = await db.query(categoryQuery, params);
  parentCategoryList = await decodeAndParseFields(parentCategoryList);

  return sendResponse(res, 200, parentCategoryList);
};

/** Function to delete Category */
export const deleteCategory = async (req, res) => {
  const { id } = req.params;
  const deleteRecord = await deleteSettingRecord("category", id);
  if (deleteRecord) {
    await insertActivityLog(req.user.sessionid, "delete", "Category", id);
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};
