import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  pointsCalculate,
  searchConditionRecord,
  storeError,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorAuditProtocol from "../sequelize/ContractorAuditProtocolSchema.js";
import ContractorAuditRecord from "../sequelize/ContractorAuditRecordSchema.js";

/**Function to create and update Contractor Audit Protocol  */
export const createUpdateContractorAuditProtocol = async (req, res) => {
  const { id, scheduled_audit, organization, department } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  // Scheduled_audit instead of contractor_audit_protocol

  /** Check that for this audit Scheduled audit protocol already exist or not */
  const [checkContractorAuditProtocolWithOrganization] = await db.query(
    "SELECT id FROM contractor_audit_protocol WHERE scheduled_audit = ? AND deleted = '0' AND id != ? AND organization = ?",
    [scheduled_audit, id, organizationId]
  );
  if (checkContractorAuditProtocolWithOrganization.length > 0) {
    return sendResponse(
      res,
      400,
      `Record with Contractor Scheduled Audit ${scheduled_audit} already exists.`
    );
  }

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Check that this organization have that particular Contractor Scheduled Audit or not*/
    const [checkScheduledAuditWithOrganization] = await db.query(
      "SELECT scheduled_audit FROM contractor_audit_protocol WHERE scheduled_audit = ? AND deleted = '0' AND id != ? AND organization = ?",
      [scheduled_audit, id, organizationId]
    );
    if (checkScheduledAuditWithOrganization.length > 0) {
      return sendResponse(
        res,
        400,
        `Record with Contractor Scheduled Audit ${scheduled_audit} already exists.`
      );
    }

    const { query, values } = updateQueryBuilder(
      ContractorAuditProtocol,
      req.body
    );
    await db.query(query, values);
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "update",
      "Contractor Audit Protocol",
      `This user Update Contractor Audit Protocol which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    /** Check if this organization already has the particular scheduled_audit */
    const [checkScheduledAuditWithOrganization] = await db.query(
      `SELECT scheduled_audit FROM contractor_audit_protocol WHERE scheduled_audit = ? AND deleted = '0' AND organization = ?`,
      [scheduled_audit, organizationId]
    );
    if (checkScheduledAuditWithOrganization.length > 0) {
      return sendResponse(
        res,
        400,
        `Record with Contractor Scheduled Audit ${scheduled_audit} already exists.`
      );
    }
    req.body.created_by = req.user.sessionid;
    req.body.status = req.body.status ?? "Scheduled";
    /** Insert Contractor Audit Protocol details */
    const { query, values } = createQueryBuilder(
      ContractorAuditProtocol,
      req.body
    );

    await db.query(query, values);

    /** Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Contractor Audit Protocol",
      `This user created a new Audit Protocol for audit scheduled ${scheduled_audit} for organization ${organizationId}`
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Contractor Audit Protocol */
export const viewAllContractorAuditProtocol = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_audit_protocol",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "contractor_audit_scheduling.name",
    "contractor_audit_protocol.status",
    "contractor_audit_scheduling.audit_type",
    "audit_template.name",
    "organization.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_audit_protocol.created_by",
    },
    {
      type: "left",
      targetTable: "contractor_audit_scheduling",
      onCondition:
        "contractor_audit_scheduling.id = contractor_audit_protocol.scheduled_audit",
    },
    {
      type: "left",
      targetTable: "audit_template",
      onCondition:
        "audit_template.id = contractor_audit_protocol.audit_protocol",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_audit_protocol.organization",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition:
        "contractor_registration.id = contractor_audit_protocol.contractor",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const contractorAuditProtocolDataFetchQuery = `SELECT contractor_audit_protocol.*, 
    contractor_registration.contractor_name as contractor_name,
    contractor_registration.cipc_registration_number AS contractor_cipc_registration_number,
    contractor_audit_scheduling.planned_audit_start_date_time, 
    contractor_audit_scheduling.planned_audit_end_date_time, contractor_audit_scheduling.audit_type, 
    contractor_audit_scheduling.repeats , contractor_audit_scheduling.no_of_cycle,
    contractor_audit_scheduling.recurrence_cycle,
    contractor_audit_protocol.created_by as created_by_id,
    audit_template.name as audit_template_name,
    contractor_audit_scheduling.name as scheduled_audit_name,
    users.name as created_by_name,contractor_audit_protocol.organization,organization.name as organization_name
    FROM contractor_audit_protocol  
    ${joinsRecord} 
    where contractor_audit_protocol.deleted = 0 ${searchCondition} ${condition}`;

  const [contractorAuditProtocolDataFetch] = await db.query(
    contractorAuditProtocolDataFetchQuery
  );

  /**Count all Contractor Audit Protocol */
  const totalRecord = await countQueryCondition(
    contractorAuditProtocolDataFetchQuery
  );

  return sendResponse(res, 200, contractorAuditProtocolDataFetch, totalRecord);
};

/**Function to delete a specific Contractor Audit Protocol */
export const deleteContractorAuditProtocol = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("contractor_audit_protocol", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "delete",
      "Contractor Audit Protocol",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/** function for adding audit record using audit template */
export const saveAuditRecord = async (req, res) => {
  const { id, organization, department, sections, audit_execution } = req.body;

  /** Check for audit execution is already executed or not  (by checking status)*/

  const [checkContractorAuditProtocol] = await db.query(
    "SELECT id, status FROM contractor_audit_protocol WHERE id = ? AND deleted = 0",
    [audit_execution]
  );

  if (checkContractorAuditProtocol[0]?.status == "Executed") {
    return sendResponse(
      res,
      400,
      "Contractor Audit Protocol is already executed"
    );
  }

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  console.log('sections: ', sections);
  const points = await pointsCalculate(sections);
  req.body.total_point = points.totalPoints;
  req.body.earned_points = points.pointsEarned;
  req.body.audit_record = req.body.sections;
  if (id) {
    const { query, values } = updateQueryBuilder(
      ContractorAuditRecord,
      req.body
    );
    await db.query(query, values);
    return sendResponse(res, 200, "Record updated successfully");
  }
  req.body.created_by = req.user.sessionid;
  const { query, values } = createQueryBuilder(ContractorAuditRecord, req.body);
  console.log(query, values);
  const [createAuditRecord] = await db.query(query, values);
  if (createAuditRecord.insertId > 0) {
    /** update status of audit execution */
    console.log(audit_execution);
    const updateContractorAuditProtocolQuery = `UPDATE contractor_audit_protocol SET status=? WHERE id=? AND deleted = 0`;
    await db.query(updateContractorAuditProtocolQuery, [
      "Executed",
      audit_execution,
    ]);

    /** update status of audit Scheduled */
    if (checkContractorAuditProtocol[0]?.scheduled_audit) {
      const updateAuditSchedulingQuery = `UPDATE contractor_audit_scheduling SET status=? WHERE id=? AND deleted = 0`;
      await db.query(updateAuditSchedulingQuery, [
        "Executed",
        checkContractorAuditProtocol[0]?.scheduled_audit,
      ]);
    }

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Contractor Audit Protocol Record",
      createAuditRecord.insertId
    );
    return sendResponse(res, 200, "Record created successfully");
  } else {
    return sendResponse(res, 200, "Record not created");
  }
};

/**Function to view all Audit Record */
export const viewAllAuditRecord = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_audit_record",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    columnName: "audit_execution",
  });

  const searchTableName = [
    "contractor_audit_record.name",
    "contractor_audit_record.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_audit_record.created_by",
    },
    {
      type: "left",
      targetTable: "contractor_audit_protocol",
      onCondition:
        "contractor_audit_protocol.id = contractor_audit_record.audit_execution",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_audit_record.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all Contractor Audit Protocol */
  const contractorAuditProtocolDataFetchQuery = `SELECT 
    contractor_audit_record.*, contractor_audit_record.created_by as created_by_id, users.name as created_by_name,organization.name as organization_name
    FROM contractor_audit_record  
    ${joinsRecord} 
    where contractor_audit_record.deleted = 0 ${searchCondition} ${condition}`;

  const [contractorAuditProtocolDataFetch] = await db.query(
    contractorAuditProtocolDataFetchQuery
  );
  contractorAuditProtocolDataFetch.forEach((element) => {
    element.audit_record = JSON.parse(element.audit_record);
  });
  /**Count all Contractor Audit Protocol */
  const totalRecord = await countQueryCondition(
    contractorAuditProtocolDataFetchQuery
  );

  return sendResponse(res, 200, contractorAuditProtocolDataFetch, totalRecord);
};
