import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  createQueryBuilder,
  whereCondition,
  makeJoins,
  countQueryCondition,
  searchConditionRecord,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  encodeAndStringifyFields,
  insertActivityLog,
  getRecord,
  getBcpComment,
  getUserListByIds,
  getEmployeeDetails,
  uniqueIdGenerator,
  getCrisisManagementJSONData,
} from "../helper/general.js";
import CrisisManagement from "../sequelize/CrisisManagementSchema.js";
import BcpTesting from "../sequelize/BcpTestingSchema.js";
import { updateQueryBuilder } from "../helper/queryBuilder.js";
import BcpManagementComment from "../sequelize/BcpManagementCommentSchema.js";

export const createUpdateCrisisManagement = async (req, res) => {
  // req.body = (await decodeAndParseFields([req.body]))[0];
  const { id, department } = req.body;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  req.body = await encodeAndStringifyFields(req.body);

  // add unique_id if id is not present
  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "CM",
      "crisis_management",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(CrisisManagement, req.body)
    : createQueryBuilder(CrisisManagement, req.body);
  const [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Crisis Management",
    `This user ${status} a new Crisis Management Record`
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

const userFields = [
  "crisis_management_team_leader",
  "deputy_crisis_management_team_leader",
  "emergency_response_coordinator",
  "technical_recovery_lead",
  "communication_officer",
  "legal_and_compliance_officer",
  "hr_representative",
  "operation_recovery_lead",
  "finance_and_administration_lead",
  "facilities_and_physical_infrastructure",
  "supply_chain_and_procurement_lead",
  "external_liaison_officer",
  "security_lead",
  "risk_management_officer",
  "crisis_documentation_officer",
  "escalation_user_id",
  "deactivation_user_id",
];

export const getCrisisManagement = async (req, res) => {
  const { id } = req.params;

  // Prepare conditions
  const condition = await whereCondition({
    table: "crisis_management",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "crisis_management.crisis_name",
    "crisis_management.crisis_occurrence_date",
    "crisis_type.name",
    "  CONCAT(declared_by.name, ' ', declared_by.surname)",
    "  bcp_management.plan_name",
    "  affected_area.name",
    "  critical_system_affected.name"
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = crisis_management.created_by",
    },
    {
      type: "left",
      targetTable: "users as declared_by",
      onCondition: "declared_by.id = crisis_management.crisis_declared_by",
    },
    {
      type: "left",
      targetTable: "roles as declared_by_role",
      onCondition: "declared_by_role.id = declared_by.role",
    },
    {
      type: "left",
      targetTable: "department as declared_by_department",
      onCondition: "declared_by_department.id = declared_by.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = crisis_management.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = crisis_management.department",
    },
    {
      type: "left",
      targetTable: "crisis_type",
      onCondition: "crisis_type.id = crisis_management.crisis_type",
    },
    {
      type: "left",
      targetTable: "bcp_management",
      onCondition: "bcp_management.id = crisis_management.bcp_management_id",
    },
    {
      type: "left",
      targetTable: "location as crisis_location",
      onCondition: "crisis_location.id = crisis_management.crisis_location",
    },
    {
      type: "left",
      targetTable: "currency",
      onCondition: "currency.id = crisis_management.currency",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = crisis_management.command_center_location",
    },
    {
      type: "left",
      targetTable: "critical_system_affected",
      onCondition:
        "critical_system_affected.id = crisis_management.critical_system_affected",
    },
    {
      type: "left",
      targetTable: "affected_area",
      onCondition: "affected_area.id = crisis_management.affected_areas",
    },
  ];

  const joinCondition = await makeJoins(joins);

  // Base query for Crisis Management
  const crisisManagementQuery = `
  SELECT 
    crisis_management.*,
    bcp_management.plan_name AS bcp_management_plan_name,
    bcp_management.plan_version_number AS bcp_management_plan_version_number,
    bcp_management.crisis_management_team_leader AS bcp_management_crisis_management_team_leader,    
    crisis_type.name AS crisis_type_name,
    crisis_type.description AS crisis_type_description,
    crisis_location.name AS crisis_location_name,
    crisis_location.description AS crisis_location_description,
    currency.name AS currency_name,
    currency.description AS currency_description,
    location.name AS command_center_location_name,
    location.description AS command_center_location_description,  
    critical_system_affected.name AS critical_system_affected_name,
    critical_system_affected.description AS critical_system_affected_description,
    affected_area.name AS affected_area_name,
    affected_area.description AS affected_area_description,
    CONCAT(declared_by.name, ' ', declared_by.surname) AS declared_by_name, 
    declared_by.profile AS declared_by_profile,
    declared_by.role as declared_by_role_id,
    declared_by_role.name as declared_by_role_name,
    declared_by_department.name as declared_by_department_name,
    organization.name AS organization_name,
    department.name AS department_name,
    users.name AS created_by_name,
    users.surname AS created_by_surname,
    users.profile AS created_by_profile


  FROM crisis_management
  ${joinCondition}

  WHERE crisis_management.deleted = 0
  ${searchCondition}
  ${condition}`;

  // Execute base query
  let [CrisisManagement] = await db.query(crisisManagementQuery);

  CrisisManagement = await decodeAndParseFields(CrisisManagement);

  CrisisManagement = await getCrisisManagementJSONData(CrisisManagement);
  // const [bcpManagement] = await db.query(
  //   `SELECT * FROM bcp_management where id IN(${CrisisManagement.map((record) => record.bcp_management_id).join(",")})`
  // );

  let bcpManagement = [];
  const bcpManagementIds = CrisisManagement.map(
    (record) => record.bcp_management_id
  ).filter(Boolean);

  if (bcpManagementIds.length > 0) {
    [bcpManagement] = await db.query(
      `SELECT *, bcp_management.plan_name AS name FROM bcp_management WHERE id IN (${bcpManagementIds.join(
        ","
      )})`
    );
  }

  const { leaderDataMap, all_members, bcpManagementData } =
    await getEmployeeDetails(bcpManagement, CrisisManagement);

  const userIds = [
    ...new Set(
      CrisisManagement.flatMap((record) =>
        userFields.map((field) => record[field])
      )
    ),
  ].filter((id) => id != null); // Exclude nulls
  // Fetch user data in a single query
  let users = [];

  if (userIds.length > 0) {
    const userQuery = `
            SELECT 
              users.id AS user_id,
              CONCAT(users.name, ' ', users.surname) AS name,
              users.profile AS profile,
              users.phone AS phone_number,
              department.name AS department,
              roles.name AS role,
              users.email AS email_address 
            FROM users
            LEFT JOIN roles ON roles.id = users.role
            LEFT JOIN department ON department.id = users.department
            WHERE users.id IN (${userIds.join(",")})
            `;

    [users] = await db.query(userQuery);
  }

  const userMap = Object.fromEntries(users.map((user) => [user.user_id, user]));
  // Step 5: Dynamically enrich records with flat structure
  const enrichedRecords = CrisisManagement.map((record) => {
    const enrichedRecord = { ...record };

    userFields.forEach((field) => {
      const userId = record[field];
      const user = userMap[userId] || {};
      // console.log(`Field: ${field}, User ID: ${userId}`);

      enrichedRecord[`${field}_name`] = user.name || null;
      enrichedRecord[`${field}_profile`] = user.profile || null;
      enrichedRecord[`${field}_phone_number`] = user.phone_number || null;
      enrichedRecord[`${field}_department`] = user.department || null;
      enrichedRecord[`${field}_role`] = user.role || null;
      enrichedRecord[`${field}_email_address`] = user.email_address || null;
    });

    return enrichedRecord;
  });

  const totalRecord = await countQueryCondition(crisisManagementQuery);

  // Send response
  return sendResponse(res, 200, enrichedRecords, totalRecord);

  // Batch process employee fields
  const employeeFieldIds = [];
  const employeeFieldMap = {};

  for (let item of CrisisManagement) {
    for (let key of allEmployeeFields) {
      if (item[key]) {
        employeeFieldIds.push(item[key]);
        employeeFieldMap[item[key]] = { key, id: item[key] };
      }
    }
  }

  // Fetch employee data in a single query
  if (employeeFieldIds.length > 0) {
    const uniqueIds = [...new Set(employeeFieldIds)];
    const leaderQuery = `
        SELECT 
          users.id AS user_id,
          CONCAT(users.name, ' ', users.surname) AS name,
          users.profile AS profile,
          users.phone AS phone_number,
          department.name AS department,
          roles.name AS role,
          users.email AS email_address 
        FROM users 
        LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id = users.department 
        WHERE users.id IN (${uniqueIds.join(",")})
      `;

    const [leaderData] = await db.query(leaderQuery);

    // Map employee data by user_id
    const leaderDataMap = leaderData.reduce((map, leader) => {
      map[leader.user_id] = leader;
      return map;
    }, {});

    // Merge leader data back into CrisisManagement
    for (let i = 0; i < CrisisManagement.length; i++) {
      const item = CrisisManagement[i];
      for (let key of allEmployeeFields) {
        if (item[key] && leaderDataMap[item[key]]) {
          const leader = leaderDataMap[item[key]];
          CrisisManagement[i] = {
            ...CrisisManagement[i],
            [`${key}_name`]: leader.name || null,
            [`${key}_phone_number`]: leader.phone_number || null,
            [`${key}_department`]: leader.department || null,
            [`${key}_role`]: leader.role || null,
            [`${key}_email_address`]: leader.email_address || null,
            [`${key}_profile`]: leader.profile || null,
          };
        }
      }
    }
  }
};
export const deleteCrisisManagement = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("crisis_management", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "crisis_management",
    `This user deleted a new crisis_management Record for organization ${id}`
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
