import moment from "moment";
import db from "../db-config.js";
import {
  insertActivityLog,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  getRecord,
  decodeAndParseFields,
  saveEquipmentHistory,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import FireEquipmentStatusHistory from "../sequelize/fireEquipmentStatusHistorySchema.js";
import FireEquipmentRegister from "../sequelize/FireEquipmentRegisterSchema.js";

const excludedFields = ["created_at", "updated_at", "created_by", "updated_by"];

/**Function to create/update Fire Equipment Register   */
export const FireEquipmentRegisterCreateUpdate = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { id, organization, department } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;

  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**If id comes in body then it will update the query */
  if (id) {
    const [oldData] = await db.query(`SELECT * FROM fire_equipment_register WHERE id = ?`, [id]);

    req.body.updated_by = req.user.sessionid;
    /**Update Fire Equipment Register Query */
    const { query, values } = updateQueryBuilder(FireEquipmentRegister, req.body);
    await db.query(query, values);

    const [newData] = await db.query(`SELECT * FROM fire_equipment_register WHERE id = ?`, [id]);

    const changedFields = {};
    for (const key in oldData[0]) {
      if (!excludedFields.includes(key) && oldData[0][key] !== newData[0][key]) {
        // changedFields[key] = { old: oldData[0][key], new: newData[0][key] };
        changedFields[key] = newData[0][key];
      }
    }
    await saveEquipmentHistory(
      "UPDATED",
      department,
      organizationId,
      "Equipment Register Updated",
      req.user.sessionid,
      id,
      null,
      changedFields
    );
    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "FireEquipmentRegister", id);
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "REI",
      "fire_equipment_register",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;
    /**Insert record for Fire Equipment Register */
    const { query, values } = createQueryBuilder(FireEquipmentRegister, req.body);
    const [createFireEquipmentRegister] = await db.query(query, values);
    // console.log("createFireEquipmentRegister: ", createFireEquipmentRegister);

    await saveEquipmentHistory(
      "CREATED",
      department,
      organizationId,
      "Equipment Register Created with " + req.body.status + " status.",
      req.user.sessionid,
      createFireEquipmentRegister.insertId
    );

    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "create", "Equipment Register", createFireEquipmentRegister.insertId);
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Fire Equipment Register */
export const viewFireEquipmentRegister = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "fire_equipment_register",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = ["CONCAT(users.name , ' ' , users.surname)", "organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = fire_equipment_register.created_by",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = fire_equipment_register.location",
    },
    {
      type: "left",
      targetTable: "equipment",
      onCondition: "equipment.id = fire_equipment_register.equipment_type",
    },
    {
      type: "left",
      targetTable: "department as responsible_department",
      onCondition: "responsible_department.id = fire_equipment_register.responsible_department",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = fire_equipment_register.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = fire_equipment_register.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);
  /**Record of all Fire Equipment Register */
  const FireEquipmentRegisterQuery = `
    SELECT fire_equipment_register.*, 
    location.name AS location_name, 
    department.name AS department_name, 
    responsible_department.name as responsible_department_name,  
    equipment.name AS equipment_type_name, 
    users.name AS created_by, 
    organization.name AS organization_name 
      FROM fire_equipment_register 
      ${joinsRecord} where fire_equipment_register.deleted = 0  ${searchCondition} ${condition}`;
  let [FireEquipmentRegister] = await db.query(FireEquipmentRegisterQuery);

  for (let register of FireEquipmentRegister) {
    const maintenanceQuery = `SELECT * FROM fire_equipment_maintenance_inspection WHERE equipment_id = ${register.id} ORDER BY created_at DESC LIMIT 1`;
    let [maintenance] = await db.query(maintenanceQuery);
    if (
      maintenance.length > 0 &&
      moment(register.next_maintenance_date).format("YYYY-MM-DD") >= moment().format("YYYY-MM-DD")
    ) {
      register.scheduled_status = "Scheduled";
      register.equipment_maintenance_inspection_id = maintenance[0]?.id;
    } else {
      register.scheduled_status = "Pending";
    }
    register.last_maintenance_date = moment(maintenance[0]?.last_maintenance_date).format("YYYY-MM-DD");
  }
  // FireEquipmentRegister = await Promise.all(
  //   FireEquipmentRegister.map(async (record) => {
  //     return {
  //       ...record,
  //       schedule: JSON.parse(record?.schedule || "[]"),
  //     };
  //   })
  // );

  /**Count of all Fire Equipment Register */
  const totalRecord = await countQueryCondition(FireEquipmentRegisterQuery);
  return sendResponse(res, 200, FireEquipmentRegister, totalRecord);
};

/**Function to delete a specific Fire Equipment Register */
export const deleteFireEquipmentRegister = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("fire_equipment_register", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Fire Equipment Register", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

export const updateFireEquipmentRegisterStatus = async (req, res) => {
  let { id, status, department, notify_to, reported_by, remarks_comments, last_updated_date, organization } = req.body;

  console.log(reported_by);

  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    organization = organizationId;
  }

  if (!id) return sendResponse(res, 400, "Id is required");
  if (!status) return sendResponse(res, 400, "status is required");

  const [checkExist] = await getRecord("fire_equipment_register", "id", id);
  // console.log(checkExist);
  if (!checkExist) return sendResponse(res, 400, "equipment does not exist with this Id");
  const { query, values } = updateQueryBuilder(FireEquipmentRegister, req.body);

  await db.query(query, values);

  const changedFields = { status };

  await saveEquipmentHistory(
    "STATUS CHANGED",
    department,
    organizationId,
    "Equipment Register status changed",
    req.user.sessionid,
    id,
    null,
    changedFields,
    last_updated_date,
    remarks_comments,
    reported_by,
    notify_to
  );

  // const { query: HistoryQuery, values: HistoryValues } = createQueryBuilder(FireEquipmentStatusHistory, {
  //   equipment_id: id,
  //   updated_date: last_updated_date,
  //   comments: remarks_comments,
  //   reported_by,
  //   notify_to,
  //   status,
  //   created_by: req.user.sessionid,
  //   department,
  //   organization,
  // });

  // await db.query(HistoryQuery, HistoryValues);

  /**Insert Activity  */
  await insertActivityLog(req.user.sessionid, "update", "FireEquipmentRegister", id);
  return sendResponse(res, 200, "Record updated successfully");
};

/**Function to view all and single Fire Equipment Register status History*/
export const viewFireEquipmentRegisterStatusHistory = async (req, res) => {
  const { id } = req.params;
  const orderBy = req.query.orderBy;
  const condition = await whereCondition({
    table: "fire_equipment_status_history",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    orderBy,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = ["organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = fire_equipment_status_history.created_by",
    },
    {
      type: "left",
      targetTable: "users AS notify_user",
      onCondition: "notify_user.id = fire_equipment_status_history.notify_to",
    },
    {
      type: "left",
      targetTable: "users AS reported_user",
      onCondition: "reported_user.id = fire_equipment_status_history.reported_by",
    },
    {
      type: "left",
      targetTable: "fire_equipment_register",
      onCondition: "fire_equipment_register.id = fire_equipment_status_history.equipment_id",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = fire_equipment_status_history.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = fire_equipment_status_history.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Fire Equipment Register */
  const FireEquipmentRegisterQuery = `SELECT fire_equipment_status_history.*,fire_equipment_register.equipment_name AS fire_equipment_name,fire_equipment_register.unique_equipment_id AS unique_equipment_id,fire_equipment_register.equipment_category AS fire_equipment_category, department.name AS department_name, users.name AS created_by, 
  organization.name AS organization_name, CONCAT(notify_user.name,' ',notify_user.surname) AS notify_to_name,CONCAT(reported_user.name,' ',reported_user.surname) AS reported_by_name
      FROM fire_equipment_status_history 
      ${joinsRecord} where fire_equipment_status_history.deleted = 0  ${searchCondition} ${condition}`;

  let [FireEquipmentRegister] = await db.query(FireEquipmentRegisterQuery);

  FireEquipmentRegister.forEach((element) => {
    element.changed_fields = JSON.parse(element?.changed_fields || "[]");
    element.created_at = moment(element.created_at).format("YYYY-MM-DD");
    element.updated_at = moment(element.updated_at).format("YYYY-MM-DD");
    element.updated_date = element.updated_date ? moment(element.updated_date).format("YYYY-MM-DD") : null;
  });

  /**Count of all Fire Equipment Register */
  const totalRecord = await countQueryCondition(FireEquipmentRegisterQuery);
  return sendResponse(res, 200, FireEquipmentRegister, totalRecord);
};

export const updateFireEquipmentRegisterHistoryStatus = async (req, res) => {
  const { ids, status } = req.body;
  if (!ids || ids?.length == 0) {
    return sendResponse(res, 400, "Ids must be an array and cannot be empty.");
  }
  const updateQuery = `UPDATE fire_equipment_register SET status = ? WHERE id IN (${ids})`;
  await db.query(updateQuery, [status]);

  const changedFields = { status };
  for (let id of ids) {
    await saveEquipmentHistory(
      "STATUS CHANGED",
      null,
      null,
      "Equipment Register status changed",
      req.user.sessionid,
      id,
      null,
      changedFields
    );
  }

  return sendResponse(res, 200, "Status updated successfully");
};
