import db from "../db-config.js";
import {
  insertActivityLog,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  searchConditionRecord,
  decodeAndParseFields,
  getRecord,
  uniqueIdGenerator,
  getOrganizationAccordingToDepartment,
  getOperationalRiskData,
  getOperationalRecordArray,
  generateOperationalRiskTable,
  sendDynamicEmail,
} from "../helper/general.js";
import OperationalRisk from "../sequelize/OperationalRiskSchema.js";
import OperationalRiskIdentification from "../sequelize/OperationalRiskIdentificationSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import WorkArea from "../sequelize/WorkAreaSchema.js";
import Activities from "../sequelize/ActivitiesSchema.js";
import sequelize from "../sequelize/sequelize.js";
import { and, Op } from "sequelize";

async function sendOperationalRiskCreationEmail(id) {
  const [operationalData] = await getOperationalRiskData(id);
  // console.log('operationalData: ', operationalData);
  const templateFileUrl = "mail_for_operational_risk_creation_template.html";
  const templateName = "Operational Risk Register Creation";
  let recordArray = await getOperationalRecordArray({
    operationalData: operationalData,
    templateFileUrl,
    templateName,
  });
  const tableHtml = generateOperationalRiskTable(operationalData);
  recordArray.operational_work_area = tableHtml;

  const usersToSend = [
    {
      name: recordArray.risk_register_approver,
      email: operationalData.approver_email,
    },
    {
      name: recordArray.risk_assessment_facilitator,
      email: operationalData.facilitator_email,
    },
  ];
  for (let user of usersToSend) {
    recordArray.name = user.name;
    sendDynamicEmail({
      to: user.email,
      subject: `Successful Creation of Risk Register - ${operationalData.risk_register_name}`,
      data: recordArray,
    });
  }
}
/**Function to create/update operational risk */
export const operationalRiskRegisterCreateUpdate = async (req, res) => {
  const { id, status, organization, department } = req.body;
  const messageStatus = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  /** Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  if (!id) {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "OPE",
      "operational_risk",
      "unique_id",
      "unique_id"
    );
    const type = "reference";
    const reference_no = await uniqueIdGenerator(
      organizationId,
      department,
      "OPE",
      "operational_risk",
      "reference_no",
      type
    );
    req.body.unique_id = unique_id;
    req.body.reference_no = reference_no;
  }
  const allWorkAreas = [];
  const allActivities = [];
  const allIdentification = [];
  const transaction = await sequelize.transaction();
  if (id) {
    /** add a check whether operational risk is rejected before updating */
    const [statusCheck] = await getRecord("operational_risk", "id", id);
    if (statusCheck.approval_status == "Rejected") {
      return sendResponse(res, 400, "Record already rejected !");
    }
  }
  // Step 1: Create or update the main OperationalRisk record
  const [operationalRisk] = await OperationalRisk.upsert(
    {
      id, // This is the existing id or undefined for a new record
      organization: req.body.organization,
      risk_register_name: req.body.risk_register_name,
      assessment_date: req.body.assessment_date,
      review_date: req.body.review_date,
      risk_context: req.body.risk_context,
      risk_assessment_facilitator:
        req.body?.risk_assessment_facilitator || null,
      risk_register_approver: req.body?.risk_register_approver || null,
      register_status: req.body.register_status,
      main_process: req.body.main_process,
      meeting_id: req.body?.meeting_id || null,
      unique_id: req.body.unique_id,
      reference_no: req.body.reference_no,
      department: req.body.department,
      created_by: req.body.created_by,
      updated_by: req.body.updated_by,
    },
    { transaction, returning: true }
  );

  if (id && statusCheck.status == "Draft" && req.body.status == "Complete") {
    const [operationalData] = await getOperationalRiskData(id);

    const dataToSend = await getOperationalRecordArray({
      operationalData,
      templateFileUrl: "mail_for_operational_risk_approval_template.html",
      templateName: "Operational Risk Register Approval",
    });

    dataToSend.name = dataToSend.risk_register_approver;
    sendDynamicEmail({
      to: operationalData.approver_email,
      subject: `Risk Register Awaiting Your Approval - ${dataToSend.risk_register_name}`,
      data: dataToSend,
    });
  }
  if (
    id &&
    statusCheck.approval_status == "Pending Approval" &&
    req.body.approval_status == "Approved"
  ) {
    const [operationalData] = await getOperationalRiskData(id);

    const dataToSend = await getOperationalRecordArray({
      operationalData,
      templateFileUrl: "mail_for_operational_risk_approval_template.html",
      templateName: "Operational Risk Register Approval",
    });

    const usersToSend = [
      {
        name: `${operationalData.created_by} ${operationalData.created_by_surname}`,
        email: operationalData.created_by_email,
      },
      {
        name: dataToSend.risk_assessment_facilitator,
        email: operationalData.facilitator_email,
      },
      {
        name: dataToSend.risk_register_approver,
        email: operationalData.approver_email,
      },
    ];
    for(let user of usersToSend){
      dataToSend.name = user.name;
      sendDynamicEmail({
        to: user.email,
        subject: `Risk Register Approval - ${dataToSend.risk_register_name}`,
        data: dataToSend,
      });
    }
  }

  // Step 2: Process work areas and activities, using the OperationalRisk id
  for (const workAreaData of req.body.work_areas) {
    const [workArea] = await WorkArea.upsert(
      {
        id: workAreaData.id,
        name: workAreaData.name,
        organization: req.body.organization,
        department: req.body.department,
        created_by: req.user.sessionid,
        operational_risk_id: operationalRisk.id, // Associate WorkArea with OperationalRisk
      },
      { transaction, returning: true }
    );
    allWorkAreas.push(workArea.id);
    for (const activityData of workAreaData.activities) {
      const [activity] = await Activities.upsert(
        {
          id: activityData.id,
          name: activityData.name,
          frequency: activityData.frequency,
          work_id: workArea.id,
          organization: req.body.organization,
          department: req.body.department,
          created_by: req.user.sessionid,
          operational_risk_id: operationalRisk.id, // Associate WorkArea with OperationalRisk
        },
        { transaction, returning: true }
      );
      allActivities.push(activity.id);
      for (const riskData of activityData.risks) {
        const riskIdentification = await OperationalRiskIdentification.create(
          {
            risk_register_id: operationalRisk.id, // Link to main OperationalRisk
            name: riskData.name,
            description: riskData.description,
            category: riskData.category,
            hazard: riskData.hazard,
            impact: riskData.impact,
            frequency: riskData.frequency,
            probability: riskData.probability,
            consequences: riskData.consequences,
            risk_rating: riskData.risk_rating,
            risk_ranking: riskData.risk_ranking,
            control_assessments: JSON.stringify(riskData.control_assessments),
            residual_rating: riskData.residual_rating,
            residual_ranking: riskData.residual_ranking,
            opportunity_identification: riskData.opportunity_identification,
            department: req.body.department,
            organization: req.body.organization,
            opportunity_description: JSON.stringify(
              riskData.opportunity_description
            ),
            activity_id: activity.id,
            created_by: req.user.sessionid,
          },
          { transaction }
        );
        allIdentification.push(riskIdentification.id);
      }
    }
  }

  /** Soft Delete the record whose ids are not present in the request for all tables (in case of update only) */
  if (req.body.id) {
    if (allWorkAreas.length > 0) {
      await WorkArea.update(
        { deleted: 1 },
        {
          where: {
            [Op.and]: {
              id: { [Op.notIn]: allWorkAreas },
              operational_risk_id: operationalRisk.id,
            },
          },
          transaction,
        }
      );
    }
    if (allActivities.length > 0) {
      await Activities.update(
        { deleted: 1 },
        {
          where: {
            [Op.and]: {
              id: { [Op.notIn]: allActivities },
              operational_risk_id: operationalRisk.id,
            },
          },
          transaction,
        }
      );
    }
    if (allIdentification.length > 0) {
      await OperationalRiskIdentification.update(
        { deleted: 1 },
        {
          where: {
            [Op.and]: {
              id: { [Op.notIn]: allIdentification },
              risk_register_id: operationalRisk.id,
            },
          },
          transaction,
        }
      );
    }
  }
  const [record] = await getRecord(
    "operational_risk",
    "id",
    operationalRisk.id
  );
  /// send a notification to risk owner that Objective risk has been created
  if (status == "Complete" && record.approval_status != "Rejected") {
    const userId = req.body?.risk_register_approver;
    const pageUrl = `/risk/operational-risk-register?tab=approval-workflow`;
    if (userId) {
      await insertNotification(
        "Operational Risk",
        pageUrl,
        userId,
        "url",
        req.user.sessionid
      );
    }
  }

  await sendOperationalRiskCreationEmail(operationalRisk.id);

  await transaction.commit();
  return sendResponse(res, 200, `Record ${messageStatus} successfully`);
};

/**Function to view all or a single operational risk */
export const viewOperationalRiskRegister = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "operational_risk",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(facilitator.name , ' ' , facilitator.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(approver.name , ' ' , approver.surname)",
    "operational_risk.risk_register_name",
    "operational_risk.risk_context",
    "operational_risk.unique_id",
    "operational_risk.reference_no",
    "organization.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = operational_risk.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = operational_risk.organization",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = operational_risk.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as facilitator",
      onCondition:
        "facilitator.id = operational_risk.risk_assessment_facilitator",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = operational_risk.risk_register_approver",
    },
    {
      type: "left",
      targetTable: "main_process",
      onCondition: "main_process.id = operational_risk.main_process",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  const operationalRiskQuery = `SELECT operational_risk.*, 
    main_process.name as main_process_name,
    organization.name as organization_name, users.name as created_by, users.surname as created_by_surname,
    facilitator.name as facilitator_name, facilitator.surname as facilitator_surname,
    approver.name as approver_name, approver.surname as approver_surname,
    meeting.meeting_title as meeting_name
    FROM operational_risk 
    ${joinsRecord} WHERE operational_risk.deleted = 0 ${searchCondition} ${condition}`;
  let [operationalRisk] = await db.query(operationalRiskQuery);

  operationalRisk = await decodeAndParseFields(operationalRisk);

  // Prepare the structured data
  for (let risk of operationalRisk) {
    // Fetch associated work areas
    const [workAreas] = await db.query(
      `SELECT * FROM operational_work_area WHERE operational_risk_id = ${risk.id} AND deleted = 0`
    );
    risk.work_areas = await Promise.all(
      workAreas.map(async (workArea) => {
        const [activities] = await db.query(
          `SELECT * FROM operational_activities WHERE work_id = ${workArea.id} AND deleted = 0`
        );
        workArea.activities = await Promise.all(
          activities.map(async (activity) => {
            const [risks] = await db.query(
              `SELECT operational_risk_identification.*, category.name as category_name FROM operational_risk_identification  LEFT JOIN category ON category.id = operational_risk_identification.category WHERE operational_risk_identification.activity_id = ${activity.id} AND operational_risk_identification.deleted = 0`
            );
            activity.risks = risks.map((riskData) => ({
              ...riskData,
              control_assessments: JSON.parse(
                riskData.control_assessments || "[]"
              ),
              opportunity_description: JSON.parse(
                riskData.opportunity_description || "[]"
              ),
            }));
            return {
              ...activity,
              risks: await decodeAndParseFields(activity.risks),
            };
          })
        );
        return {
          ...workArea,
          activities: await decodeAndParseFields(workArea.activities),
        };
      })
    );
  }

  const totalRecord = await countQueryCondition(operationalRiskQuery);
  return sendResponse(res, 200, operationalRisk, totalRecord);
};

/**Function to delete an operational risk */
export const deleteOperationalRiskRegister = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("operational_risk", id);
  // also delete work area and activities and identifications
  await deleteRecord("operational_work_area", id, "operational_risk_id");
  await deleteRecord("operational_activities", id, "operational_risk_id");
  await deleteRecord("operational_risk_identification", id, "risk_register_id");
  if (deleteRecordQuery) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "OperationalRisk",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Record not found");
};
