import db from "../db-config.js";
import {
  countQueryCondition,
  decodeAndParseFields,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  getStringOrTextFields,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  whereCondition,
} from "../helper/general.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import PPERepository from "../sequelize/PPERepositorySchema.js";
import UserPPERepository from "../sequelize/UserPPERepositorySchema.js";
import { createUpdateActionFunction, getCustomActionFunction } from "./customActionCreationController.js";

export const createUpdatePpeRepository = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { id, department, task_data, action_taken, sidebar_id = 309 } = req.body;

  let organizationId = req.body.organization;

  const data = {
    task_data,
    department,
    files: req.files,
    user: req.user,
    id: action_taken,
    organization: organizationId,
    sidebar_id,
  };

  if (id && req.body?.task_data && req.body?.task_data[0] && req.body?.task_data[0]?.role) {
    const { status, arr } = await createUpdateActionFunction(data, req);
    if (status) {
      const action_taken = arr[0].id;
      req.body.action_taken = action_taken;
    }
  }

  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }
  if (id) {
    req.body.status = "Scheduled";
  }
  // generate unique id if id not present
  if (!id) {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "PPE",
      "ppe_repository",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }
  req.body = await encodeAndStringifyFields(req.body);
  // console.log(req.body, "encode req.body");
  const result = id ? await PPERepository.update(req.body, { where: { id } }) : await PPERepository.create(req.body);

  await insertActivityLog(req.user.sessionid, status, "PPE Repository", id ? id : result.id);

  return sendResponse(res, 200, `Record ${status} successfully`);
};

// export const getPpeRepository = async (req, res) => {
//   const { id } = req.params;
//   const condition = await whereCondition({
//     table: "ppe_repository",
//     page: req.query.page,
//     all: req.query.all,
//     pageSize: req.query.pageSize,
//     filter: req.query.filter,
//     id,
//     user: req.user,
//     grouped: req.query.grouped,
//   });

//   const searchableFields = getStringOrTextFields(PPERepository);

//   const searchTableName = [...searchableFields];

//   const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

//   const joins = [
//     {
//       type: "left",
//       targetTable: "organization",
//       onCondition: "organization.id = ppe_repository.organization",
//     },
//     {
//       type: "left",
//       targetTable: "users",
//       onCondition: "users.id = ppe_repository.created_by",
//     },
//     {
//       type: "left",
//       targetTable: "ppe_type",
//       onCondition: "ppe_type.id = ppe_repository.type",
//     },
//     {
//       type: "left",
//       targetTable: "location",
//       onCondition: "location.id = ppe_repository.location",
//     },
//   ];

//   const joinCondition = await makeJoins(joins);

//   const ppeRepositoryQuery = `SELECT ppe_repository.*, ppe_type.name as ppe_type_name, location.name as location_name, organization.name as organization_name, CONCAT(users.name , ' ' , users.surname )  as created_by_name , users.profile AS created_by_profile FROM ppe_repository ${joinCondition} WHERE ppe_repository.deleted = 0 ${searchCondition} ${condition}`;

//   let [ppeRepository] = await db.query(ppeRepositoryQuery);

//   ppeRepository = await decodeAndParseFields(ppeRepository);

//   for (let repo of ppeRepository) {
//     // action taken
//     if (repo?.action_taken) {
//       const data = {
//         params: { id: repo?.action_taken },
//         user: req.user,
//         query: {},
//       };
//       const customActionData = await getCustomActionFunction(data);
//       repo.task_data = customActionData;
//     }
//   }

//   const totalRecord = await countQueryCondition(ppeRepositoryQuery);

//   return sendResponse(res, 200, ppeRepository, totalRecord);
// };

export const deletePpeRepository = async (req, res) => {
  const { id } = req.params;
  await PPERepository.update({ deleted: 1 }, { where: { id } });

  await insertActivityLog(req.user.sessionid, "Deleted", "PPE Repository", id);

  return sendResponse(res, 200, `Record deleted successfully`);
};

export const assignPpeRepository = async (req, res) => {
  const { userIds, ppe_id } = req.body;
  const [ppeData] = await db.query(`SELECT quantity FROM ppe_repository WHERE id = ? AND deleted = 0`, [ppe_id]);

  if (!ppeData?.length) {
    return sendResponse(res, 400, "PPE Repository not found");
  }
  const ppe = ppeData[0];
  if (!ppe?.quantity || ppe.quantity < userIds.length) {
    return sendResponse(res, 400, `Not enough quantity available. Available quantity: ${ppe.quantity}`);
  }

  const [assignedPpe] = await db.query(
    `SELECT id FROM user_ppe_repository WHERE ppe_repository_id = ? AND user_id IN (${userIds}) AND deleted = 0`,
    [ppe_id]
  );
  if (assignedPpe?.length) {
    return sendResponse(res, 400, "PPE Repository already assigned to this user.");
  }

  for (let userId of userIds) {
    const { query, values } = createQueryBuilder(UserPPERepository, {
      ...req.body,
      user_id: userId,
      ppe_repository_id: ppe_id,
    });
    const [result] = await db.query(query, values);
    await insertActivityLog(req.user.sessionid, "Assigned", "UserPPERepository", result.insertId);
  }
  const updatedQuantity = ppe.quantity - userIds.length;

  const [updatedPpe] = await db.query(`UPDATE ppe_repository SET quantity = ${updatedQuantity} WHERE id = ${ppe_id}`);

  return sendResponse(res, 200, `Record assigned successfully`);
};

export const getAssignedPpeRepository = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "ppe_repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const joins = [
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = ppe_repository.organization",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = ppe_repository.created_by",
    },
    {
      type: "left",
      targetTable: "ppe_type",
      onCondition: "ppe_type.id = ppe_repository.type",
    },
    {
      type: "left",
      targetTable: "user_ppe_repository",
      onCondition: "user_ppe_repository.ppe_repository_id = ppe_repository.id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const query = `
           SELECT 
ppe_repository.id,
    ppe_repository.name,
    ppe_repository.quantity,
    users.id as user_id,
    users.name,
    users.email
FROM 
    ppe_repository
 JOIN 
    user_ppe_repository ON ppe_repository.id = user_ppe_repository.ppe_repository_id
 JOIN 
    users ON user_ppe_repository.user_id = users.id
ORDER BY 
    ppe_repository.id, users.id;
  `;

  const [result] = await db.query(query);
  console.log("result: ", result);

  const ppeRepositoryMap = {};

  result.forEach((row) => {
    const ppeRepositoryId = row.id;

    if (!ppeRepositoryMap[ppeRepositoryId]) {
      ppeRepositoryMap[ppeRepositoryId] = {
        id: row.id,
        repository_name: row.name,
        quantity: row.quantity,
        users: [],
      };
    }

    ppeRepositoryMap[ppeRepositoryId].users.push({
      user_id: row["user_id"],
      user_name: row["name"],
      email: row["email"],
    });
  });

  const equipmentArray = Object.values(ppeRepositoryMap);

  return sendResponse(res, 200, equipmentArray);
};

export const getPpeRepository = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "ppe_repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const page = parseInt(req.query.page) || 1;
  const pageSize = parseInt(req.query.pageSize) || 10;
  const offset = (page - 1) * pageSize;

  const searchableFields = getStringOrTextFields(PPERepository);

  const searchTableName = [...searchableFields];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = ppe_repository.organization",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = ppe_repository.created_by",
    },
    {
      type: "left",
      targetTable: "ppe_type",
      onCondition: "ppe_type.id = ppe_repository.type",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = ppe_repository.location",
    },
    {
      type: "left",
      targetTable: "user_ppe_repository",
      onCondition: "user_ppe_repository.ppe_repository_id = ppe_repository.id",
    },
    {
      type: "left",
      targetTable: "users as assigned_users",
      onCondition: "assigned_users.id = user_ppe_repository.user_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const ppeRepositoryQuery = `
    SELECT 
      ppe_repository.*,
      ppe_type.name as ppe_type_name,
      location.name as location_name,
      organization.name as organization_name,
      CONCAT(users.name, ' ', users.surname) as created_by_name,
      users.profile as created_by_profile,
      assigned_users.id as assigned_user_id,
      CONCAT(assigned_users.name, ' ', assigned_users.surname) as assigned_user_name,
      assigned_users.email as assigned_user_email
    FROM 
      ppe_repository
    ${joinCondition}
    WHERE 
      ppe_repository.deleted = 0
      ${searchCondition}
      ${condition}
    
  `;

  let [rawResults] = await db.query(ppeRepositoryQuery);
  const countQuery = `
    SELECT COUNT(DISTINCT ppe_repository.id) as total
    FROM ppe_repository
    ${joinCondition}
    WHERE ppe_repository.deleted = 0
    ${searchCondition}
    ${condition};
  `;
  const [countResult] = await db.query(countQuery);
  const totalRecord = countResult[0]?.total || 0;

  // Process results into a grouped structure
  const ppeRepositoryMap = new Map();

  for (const row of rawResults) {
    const repoId = row.id;

    // Initialize repository entry if it doesn't exist
    if (!ppeRepositoryMap.has(repoId)) {
      ppeRepositoryMap.set(repoId, {
        ...row, // Include all ppe_repository fields
        ppe_type_name: row.ppe_type_name,
        location_name: row.location_name,
        organization_name: row.organization_name,
        created_by_name: row.created_by_name,
        created_by_profile: row.created_by_profile,
        assigned_users: [], // Initialize as empty array
      });
    }

    // Add assigned user if available
    if (row.assigned_user_id) {
      ppeRepositoryMap.get(repoId).assigned_users.push({
        id: row.assigned_user_id,
        name: row.assigned_user_name,
        surname: row.assigned_user_surname,
        email: row.assigned_user_email,
      });
    }
  }

  const ppeRepository = Array.from(ppeRepositoryMap.values());

  for (let repo of ppeRepository) {
    // Add action taken data if applicable
    if (repo?.action_taken) {
      const data = {
        params: { id: repo?.action_taken },
        user: req.user,
        query: {},
      };
      const customActionData = await getCustomActionFunction(data);
      repo.task_data = customActionData;
    }
  }

  // const totalRecord = await countQueryCondition(ppeRepositoryQuery);

  return sendResponse(res, 200, ppeRepository, totalRecord);
};


export const getPpeRepositoryQuantity = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "ppe_repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchableFields = getStringOrTextFields(PPERepository);

  const searchTableName = [...searchableFields];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = ppe_repository.organization",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = ppe_repository.created_by",
    },
    {
      type: "left",
      targetTable: "ppe_type",
      onCondition: "ppe_type.id = ppe_repository.type",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = ppe_repository.location",
    },
    {
      type: "left",
      targetTable: "user_ppe_repository",
      onCondition: "user_ppe_repository.ppe_repository_id = ppe_repository.id",
    },
    {
      type: "left",
      targetTable: "users as assigned_users",
      onCondition: "assigned_users.id = user_ppe_repository.user_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const ppeRepositoryQuery = `
  SELECT 
  ppe_repository.*,
  ppe_type.name as ppe_type_name,
  location.name as location_name,
  organization.name as organization_name,
  CONCAT(users.name, ' ', users.surname) as created_by_name,
  users.profile as created_by_profile,
  assigned_users.id as assigned_user_id,
  CONCAT(assigned_users.name, ' ', assigned_users.surname) as assigned_user_name,
  assigned_users.email as assigned_user_email
  FROM 
  ppe_repository
  ${joinCondition}
  WHERE 
  ppe_repository.deleted = 0
  ${searchCondition}
  ${condition}
  `;

  let [rawResults] = await db.query(ppeRepositoryQuery);
  const countQuery = `
    SELECT COUNT(DISTINCT ppe_repository.id) as total
    FROM ppe_repository
    ${joinCondition}
    WHERE ppe_repository.deleted = 0
    ${searchCondition}
    ${condition};
  `;
  const [countResult] = await db.query(countQuery);
  const totalRecord = countResult[0]?.total || 0;

  // Process results into a grouped structure
  const ppeRepositoryMap = new Map();

  for (const row of rawResults) {
    const repoId = row.id;

    // Initialize repository entry if it doesn't exist
    if (!ppeRepositoryMap.has(repoId)) {
      ppeRepositoryMap.set(repoId, {
        ...row, // Include all ppe_repository fields
        ppe_type_name: row.ppe_type_name,
        location_name: row.location_name,
        organization_name: row.organization_name,
        created_by_name: row.created_by_name,
        created_by_profile: row.created_by_profile,
        assigned_users: [], // Initialize as empty array
      });
    }

    // Add assigned user if available
    if (row.assigned_user_id) {
      ppeRepositoryMap.get(repoId).assigned_users.push({
        id: row.assigned_user_id,
        name: row.assigned_user_name,
        surname: row.assigned_user_surname,
        email: row.assigned_user_email,
      });
    }
  }

  const ppeRepository = Array.from(ppeRepositoryMap.values());

  for (let repo of ppeRepository) {
    // Add action taken data if applicable
    if (repo?.action_taken) {
      const data = {
        params: { id: repo?.action_taken },
        user: req.user,
        query: {},
      };
      const customActionData = await getCustomActionFunction(data);
      repo.task_data = customActionData;
    }
  }

  let expandedData = [];
  for (const item of ppeRepository) {
    const quantity = Number(item.quantity) || 0;
    for (let i = 0; i < quantity; i++) {
      expandedData.push({ ...item });
    }
  }

  // 2. Pagination logic
  const page = parseInt(req.query.page) || 1;
  const pageSize = parseInt(req.query.pageSize) || 10;
  const startIndex = (page - 1) * pageSize;
  const endIndex = startIndex + pageSize;

  const paginatedData = expandedData.slice(startIndex, endIndex);
  const totalExpandedRecords = expandedData.length;
  return sendResponse(res, 200, paginatedData, totalExpandedRecords);
};