import {
  countQueryCondition,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import db from "../db-config.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import VehicleIncidentForm from "../sequelize/VehicleIncidentFormSchema.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";

/**Function to create Vehicle Incident Form  */
export const createVehicleIncidentForm = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const { id, sidebar_id = 190 } = req.body;
  const sessionId = req.user.sessionid;
  let organizationId = req.body.organization;
  const { department, comments, description_of_damage } = req.body;

  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0]?.organization;
    if (organizationId) {
      req.body.organization = organizationId;
    }
  }

  req.body.comments = await encodeSingle_statement(comments);
  req.body.description_of_damage = await encodeSingle_statement(
    description_of_damage
  );
  /** Multiple Image Handle */
  const injured_persons = req.body?.injured_persons || [];
  for (let i = 0; i <= injured_persons.length; i++) {
    if (
      (req.files && req.files[`injured_persons[${i}][alcohol_file]`]) ||
      req.body[`injured_persons[${i}][alcohol_file]`]
    ) {
      const file =
        req.files && req.files[`injured_persons[${i}][alcohol_file]`];
      if (typeof file != "string" && typeof file == "object" && file !== null) {
        // const filePath = await uploadFile("vehicle_incident_form", file);
        const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
        // req.body[`injured_persons`][i]["file"] = filePath;
        req.body[`injured_persons`][i]["alcohol_ddrm_id"] = ddrm_id;
      }
    }
    if (
      (req.files && req.files[`injured_persons[${i}][substance_file]`]) ||
      req.body[`injured_persons[${i}][substance_file]`]
    ) {
      const file =
        req.files && req.files[`injured_persons[${i}][substance_file]`];
      if (typeof file != "string" && typeof file == "object" && file !== null) {
        // const filePath = await uploadFile("vehicle_incident_form", file);
        const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
        // req.body[`injured_persons`][i]["file"] = filePath;
        req.body[`injured_persons`][i]["substance_ddrm_id"] = ddrm_id;
      }
    }
  }
  req.body.created_by = sessionId;
  const { query, values } = id
    ? updateQueryBuilder(VehicleIncidentForm, req.body)
    : createQueryBuilder(VehicleIncidentForm, req.body);
  const createVehicleForm = await db.query(query, values);
  if (!id && createVehicleForm.insertId === 0) {
    return sendResponse(res, 404, "Cannot create record");
  }

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Vehicle Incident Form",
    `This user created a new Vehicle Incident Form Record for organization ${organizationId}`
  );

  return sendResponse(res, 200, "Record created successfully");
};

export const getVehicleIncidentForm = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "vehicle_incident_form",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });
  const searchTableName = [
    "organization.name",
    "department.name",
    "CONCAT(user.name , ' ' , user.surname)",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = vehicle_incident_form.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = vehicle_incident_form.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = vehicle_incident_form.department",
    },
  ];
  const joinCondition = await makeJoins(joins);
  const vehicleIncidentFormQuery = `SELECT vehicle_incident_form.* , organization.name as organization_name ,department.name as department_name , users.name as created_by_name , users.surname as created_by_surname  , users.profile as created_by_profile FROM vehicle_incident_form ${joinCondition} WHERE vehicle_incident_form.deleted = 0 ${condition} ${searchCondition}`;
  const totalRecord = await countQueryCondition(vehicleIncidentFormQuery);

  let [vehicleIncidentForm] = await db.query(vehicleIncidentFormQuery);
  for (let i = 0; i < vehicleIncidentForm.length; i++) {
    const { injured_persons } = vehicleIncidentForm[i];
    const injured_persons_array = JSON.parse(injured_persons || "[]");
    vehicleIncidentForm[i].injured_persons = injured_persons_array;
    for (let j = 0; j < injured_persons_array.length; j++) {
      if (injured_persons_array[j].alcohol_ddrm_id) {
        const [repositoryRecordAlcohol] = await getRecord(
          "repository",
          "id",
          injured_persons_array[j]?.alcohol_ddrm_id
        );
        const [repositoryRecordSubstance] = await getRecord(
          "repository",
          "id",
          injured_persons_array[j]?.substance_ddrm_id
        );
        injured_persons_array[j].alcohol_file = repositoryRecordAlcohol?.url;
        injured_persons_array[j].substance_file =
          repositoryRecordSubstance?.url;
      }
    }
  }
  vehicleIncidentForm = await decodeAndParseFields(vehicleIncidentForm);
  return sendResponse(res, 200, vehicleIncidentForm, totalRecord);
};

export const deleteVehicleIncidentForm = async (req, res) => {
  const { id } = req.params;
  const deleteForm = await deleteRecord("vehicle_incident_form", id);
  if (deleteForm) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Injury On Duty Form",
      `This user deleted a new Injury On duty Form Record for organization ${id}`
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Cannot delete record");
};

/// images should be add in public on server
export const getVehicleCodes = async (req, res) => {
  const LICENSE_CODE = [
    { value: "A", title: "A", image: "/vehicle_icons/bike.png" },
    { value: "A1", title: "A1", image: "/vehicle_icons/cycle.png" },
    { value: "B", title: "B", image: "/vehicle_icons/car.jpeg" },
    { value: "C1", title: "C1", image: "/vehicle_icons/tow-truck.png" },
    { value: "C", title: "C", image: "/vehicle_icons/car.png" },
    { value: "EB", title: "EB", image: "/vehicle_icons/car.png" },
    { value: "EC1", title: "EC1", image: "/vehicle_icons/car.png" },
    { value: "EC", title: "EC", image: "/vehicle_icons/car.png" },
  ];
  return sendResponse(res, 200, LICENSE_CODE);
};
