import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import WasteManifest from "../sequelize/WasteManifestSchema.js";

export const createUpdateWasteManifest = async (req, res) => {
  // req.user = { sessionid: 1 };
  const { id, department } = req.body;
  let status = id ? "Updated" : "Created";
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].id;
  }

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  req.body = await encodeAndStringifyFields(req.body);

  if(!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "WM",
      "waste_manifest",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }
  
  const { query, values } = id
    ? updateQueryBuilder(WasteManifest, req.body)
    : createQueryBuilder(WasteManifest, req.body);

  const [result] = await db.query(query, values);

  await insertActivityLog(
    req.user?.sessionid,
    status,
    "Waste Manifest",
    id ? id : result.insertId
  );

  return sendResponse(res, 200, `Record ${status} Successfully`);
};

export const viewWasteManifest = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "waste_manifest",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchFields = [
    "waste_manifest.collection_date",
    "waste_manifest.disposal_date",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = waste_manifest.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = waste_manifest.organization",
    },
    {
      type: "left",
      targetTable: "waste_type",
      onCondition: "waste_type.id = waste_manifest.waste_type",
    },
    {
      type: "left",
      targetTable: "waste_disposal_method",
      onCondition:
        "waste_disposal_method.id = waste_manifest.disposable_method",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition: "contractor_registration.id = waste_manifest.contractor",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = waste_manifest.site",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const fetchQuery = `SELECT waste_manifest.*, CONCAT(users.name , ' ' , users.surname) AS created_by_name , users.profile AS created_by_profile, organization.name AS organization_name , waste_type.name AS waste_type_name , waste_disposal_method.name AS waste_disposal_method_name , contractor_registration.contractor_name AS contractor_name, location.name AS site_name FROM waste_manifest ${joinsRecord} WHERE waste_manifest.deleted = 0 ${searchCondition} ${condition}`;
  let [records] = await db.query(fetchQuery);

  records = await decodeAndParseFields(records);

  const totalRecord = await countQueryCondition(fetchQuery);

  return sendResponse(res, 200, records, totalRecord);
};

/** Function to delete WasteManifest */
export const deleteWasteManifest = async (req, res) => {
  const { id } = req.params;
  const deletedRecord = await deleteRecord(WasteManifest, id);
  if (deletedRecord) {
    await insertActivityLog(req.user.sessionid, "delete", "WasteManifest", id);
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};
