import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createVehicleIncidentForm,
  deleteVehicleIncidentForm,
  getVehicleCodes,
  getVehicleIncidentForm,
} from "../controller/vehicleIncidentFormController.js";

const vehicleIncidentFormRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Vehicle Incident Form
 *   description: API operations for managing vehicle incident forms
 */

/**
 * @swagger
 * /api/vehicleIncidentForm/create-vehicle-incident-form:
 *   post:
 *     summary: Create a new vehicle incident form
 *     tags: [Vehicle Incident Form]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/VehicleIncident'
 *           example:
 *             incident_id: 1
 *             category_id: 2
 *             organization: "Company A"
 *             date_of_incident: "2023-01-01"
 *             time_of_incident: "12:00"
 *             immediate_cause_of_incident: "Accident"
 *             name_of_road: "Main St"
 *             location: "City Center"
 *             name_of_driver: "John Doe"
 *             sap_number: "SAP123"
 *             license_no: "ABC123"
 *             restrictions_on_license: "None"
 *             year_issued: 2015
 *             code: "INC001"
 *             experience_in_current_work: "5 years"
 *             alcohol_substance_testing_done: 1
 *             substance_testing: "Breathalyzer"
 *             tested_by: "Internal"
 *             specify: "N/A"
 *             alcohol_substance_test_results: "Negative"
 *             method_of_testing: "Standard"
 *             type_of_incident_category: "Collision"
 *             was_vehicle_locked_theft: 0
 *             was_vehicle_parked_theft: 1
 *             accessories_in_car_theft: "GPS"
 *             was_car_fitted_with_tracking_device_theft: 1
 *             was_vehicle_parked_hijack: 0
 *             with_no_inside_or_near: 1
 *             was_vehicle_with_driver_inside_hijack: 0
 *             was_vehicle_in_motion_hijack: 1
 *             accessories_in_car_hijack: "None"
 *             was_car_fitted_with_tracking_device_hijack: 0
 *             car_registration: "XYZ 1234"
 *             car_make: "Toyota"
 *             car_model: "Corolla"
 *             car_year: 2020
 *             with_driver_inside_vehicle: 1
 *             kms_driven: 15000
 *             was_case_reported_to_insurer: 1
 *             was_case_reported_to_saps: 0
 *             severity_of_damage: "Major"
 *             car_owned_by: "Internal Business"
 *             specify_car_owned_by: "N/A"
 *             car_parts_damaged: "Front Bumper"
 *             estimated_cost_of_damage: 5000
 *             other_vehicle_registration_no: "ABC 5678"
 *             other_vehicle_make: "Ford"
 *             other_vehicle_driver_name: "Jane Smith"
 *             other_vehicle_owner: "Company B"
 *             other_vehicle_driver_contact: "555-1234"
 *             other_vehicle_estimated_cost_damage: 3000
 *             other_property_type: "Fence"
 *             other_property_owner: "City Council"
 *             other_property_estimated_cost_damage: 2000
 *             description_of_damage: "Front bumper damaged in collision"
 *             comments: "Witness available"
 *             form_completed_by: "John Doe"
 *             designation: "Manager"
 *             signature: "John's Signature"
 *             date: "2023-01-02"
 *     responses:
 *       200:
 *         description: Vehicle incident form successfully created
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *               example:
 *                 message: "Vehicle incident form created successfully."
 *       400:
 *         description: Invalid input
 *       500:
 *         description: Server error
 */
vehicleIncidentFormRoutes.post(
  "/create-vehicle-incident-form",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      176,
      "vehicle_incident_form"
    )(req, res, next);
  },
  controllerWrapper(createVehicleIncidentForm)
);

/**
 * @swagger
 * /api/vehicleIncidentForm/get-vehicle-incident-form/{id}:
 *   get:
 *     summary: Get a specific vehicle incident form by ID
 *     tags: [Vehicle Incident Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the vehicle incident form
 *     responses:
 *       200:
 *         description: Vehicle incident form data
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 incident_id:
 *                   type: integer
 *                 category_id:
 *                   type: integer
 *                 organization:
 *                   type: string
 *                 date_of_incident:
 *                   type: string
 *                   format: date
 *               example:
 *                 incident_id: 1
 *                 category_id: 2
 *                 organization: "Company A"
 *                 date_of_incident: "2023-01-01"
 *       404:
 *         description: Vehicle incident form not found
 */
vehicleIncidentFormRoutes.get(
  "/get-vehicle-incident-form/:id",
  verifyToken,
  sideBarPermissionCheck("view", 176, "vehicle_incident_form"),
  controllerWrapper(getVehicleIncidentForm)
);

/**
 * @swagger
 * /api/vehicleIncidentForm/delete-vehicle-incident-form/{id}:
 *   delete:
 *     summary: Delete a specific vehicle incident form by ID
 *     tags: [Vehicle Incident Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the vehicle incident form to delete
 *     responses:
 *       200:
 *         description: Vehicle incident form successfully deleted
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *               example:
 *                 message: "Vehicle incident form deleted successfully."
 *       404:
 *         description: Vehicle incident form not found
 */
vehicleIncidentFormRoutes.delete(
  "/delete-vehicle-incident-form/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 176, "vehicle_incident_form"),
  controllerWrapper(deleteVehicleIncidentForm)
);

/**
 * @swagger
 * components:
 *   schemas:
 *     VehicleIncident:
 *       type: object
 *       properties:
 *         incident_id:
 *           type: integer
 *           description: The ID of the vehicle incident of initial report
 *         category_id:
 *           type: integer
 *           description: Category ID of the incident
 *         organization:
 *           type: string
 *           description: Organization associated with the incident
 *         date_of_incident:
 *           type: string
 *           format: date
 *           description: Date of the incident
 *         time_of_incident:
 *           type: string
 *           format: time
 *           description: Time of the incident
 *         immediate_cause_of_incident:
 *           type: string
 *           description: Immediate cause of the incident
 *         name_of_road:
 *           type: string
 *           description: Name of the road where the incident occurred
 *         location:
 *           type: string
 *           description: Location of the incident
 *         name_of_driver:
 *           type: string
 *           description: Name of the driver involved in the incident
 *         sap_number:
 *           type: string
 *           description: SAP number of the driver
 *         license_no:
 *           type: string
 *           description: License number of the driver
 *         restrictions_on_license:
 *           type: string
 *           description: Restrictions on the driver's license
 *         year_issued:
 *           type: integer
 *           description: Year the driver's license was issued
 *         code:
 *           type: string
 *           description: Code associated with the incident
 *         experience_in_current_work:
 *           type: string
 *           description: Driver's experience in current work
 *         alcohol_substance_testing_done:
 *           type: integer
 *           description: Whether alcohol or substance testing was done (0 or 1)
 *           enum: [0, 1]
 *         substance_testing:
 *           type: string
 *           description: Type of substance testing done
 *         tested_by:
 *           type: string
 *           description: Whether the testing was done internally or externally
 *           enum: ["Internal", "External"]
 *         specify:
 *           type: string
 *           description: Additional specification if needed
 *         alcohol_substance_test_results:
 *           type: string
 *           description: Results of the alcohol or substance testing
 *         method_of_testing:
 *           type: string
 *           description: Method of testing used
 *         type_of_incident_category:
 *           type: string
 *           description: Type of incident category
 *         was_vehicle_locked_theft:
 *           type: integer
 *           description: Whether the vehicle was locked during theft (0 or 1)
 *           enum: [0, 1]
 *         was_vehicle_parked_theft:
 *           type: integer
 *           description: Whether the vehicle was parked during theft (0 or 1)
 *           enum: [0, 1]
 *         accessories_in_car_theft:
 *           type: string
 *           description: Accessories in the car during theft
 *         was_car_fitted_with_tracking_device_theft:
 *           type: integer
 *           description: Whether the car was fitted with a tracking device during theft (0 or 1)
 *           enum: [0, 1]
 *         was_vehicle_parked_hijack:
 *           type: integer
 *           description: Whether the vehicle was parked during hijack (0 or 1)
 *           enum: [0, 1]
 *         with_no_inside_or_near:
 *           type: integer
 *           description: Whether there was anyone inside or near the vehicle during hijack (0 or 1)
 *           enum: [0, 1]
 *         was_vehicle_with_driver_inside_hijack:
 *           type: integer
 *           description: Whether the vehicle was with the driver inside during hijack (0 or 1)
 *           enum: [0, 1]
 *         was_vehicle_in_motion_hijack:
 *           type: integer
 *           description: Whether the vehicle was in motion during hijack (0 or 1)
 *           enum: [0, 1]
 *         accessories_in_car_hijack:
 *           type: string
 *           description: Accessories in the car during hijack
 *         was_car_fitted_with_tracking_device_hijack:
 *           type: integer
 *           description: Whether the car was fitted with a tracking device during hijack (0 or 1)
 *           enum: [0, 1]
 *         car_registration:
 *           type: string
 *           description: Registration number of the car
 *         car_make:
 *           type: string
 *           description: Make of the car
 *         car_model:
 *           type: string
 *           description: Model of the car
 *         car_year:
 *           type: integer
 *           description: Year of the car
 *         with_driver_inside_vehicle:
 *           type: integer
 *           description: Whether the driver was inside the vehicle (0 or 1)
 *           enum: [0, 1]
 *         kms_driven:
 *           type: integer
 *           description: Kilometers driven
 *         was_case_reported_to_insurer:
 *           type: integer
 *           description: Whether the case was reported to the insurer (0 or 1)
 *           enum: [0, 1]
 *         was_case_reported_to_saps:
 *           type: integer
 *           description: Whether the case was reported to SAPS (0 or 1)
 *           enum: [0, 1]
 *         severity_of_damage:
 *           type: string
 *           description: Severity of the damage
 *         car_owned_by:
 *           type: string
 *           description: Car ownership details
 *         specify_car_owned_by:
 *           type: string
 *           description: Additional details about ownership
 *         car_parts_damaged:
 *           type: string
 *           description: Parts of the car that are damaged
 *         estimated_cost_of_damage:
 *           type: integer
 *           description: Estimated cost of the damage
 *         other_vehicle_registration_no:
 *           type: string
 *           description: Registration number of other vehicle involved
 *         other_vehicle_make:
 *           type: string
 *           description: Make of the other vehicle
 *         other_vehicle_driver_name:
 *           type: string
 *           description: Name of the driver of the other vehicle
 *         other_vehicle_owner:
 *           type: string
 *           description: Owner of the other vehicle
 *         other_vehicle_driver_contact:
 *           type: string
 *           description: Contact number of the driver of the other vehicle
 *         other_vehicle_estimated_cost_damage:
 *           type: integer
 *           description: Estimated cost of damage for the other vehicle
 *         other_property_type:
 *           type: string
 *           description: Type of other property involved
 *         other_property_owner:
 *           type: string
 *           description: Owner of the other property
 *         other_property_estimated_cost_damage:
 *           type: integer
 *           description: Estimated cost of damage for the other property
 *         description_of_damage:
 *           type: string
 *           description: Description of the damage
 *         comments:
 *           type: string
 *           description: Additional comments
 *         form_completed_by:
 *           type: string
 *           description: Name of the person completing the form
 *         designation:
 *           type: string
 *           description: Designation of the person completing the form
 *         signature:
 *           type: string
 *           description: Signature of the person completing the form
 *         date:
 *           type: string
 *           description: Date of form completion
 */

/**
 * @swagger
 * /api/incidentVehicle/get-vehicle-code:
 *   get:
 *     summary: Get All Vehicle Code with Images
 *     tag: [Vehicle Incident Form]
 */

vehicleIncidentFormRoutes.get(
  "/get-vehicle-code",
  verifyToken,
  controllerWrapper(getVehicleCodes)
);

export default vehicleIncidentFormRoutes;
