import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateAffectedArea,
  deleteAffectedArea,
  viewAffectedArea,
} from "../controller/affectedAreaController.js";

/**
 * @swagger
 * tags:
 *   name:  affected area
 *   description: API operations for managing  affected area
 */
const affectedAreaRoutes = Router();

/**
 * @swagger
 * /api/0/create-update-affected-area:
 *   post:
 *     summary: Create or update a  affected area
 *     tags: [ affected area]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
affectedAreaRoutes.post(
  "/create-update-affected-area",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      450,
      "affected_area"
    )(req, res, next);
  },
  controllerWrapper(createUpdateAffectedArea)
);

/**
 * @swagger
 * /api/affectedArea/get-affected-area:
 *   get:
 *     summary: Get all records of  affected area
 *     tags: [ affected area]
 */
affectedAreaRoutes.get(
  "/get-affected-area",
  verifyToken,
  sideBarPermissionCheck("view", 450, "affected_area"),
  controllerWrapper(viewAffectedArea)
);

/**
 * @swagger
 * /api/affectedArea/get-affected-area/{id}:
 *   get:
 *     summary: Get a specific record of  affected area
 *     tags: [ affected area]
 */
affectedAreaRoutes.get(
  "/get-affected-area/:id",
  verifyToken,
  sideBarPermissionCheck("view", 450, "affected_area"),
  controllerWrapper(viewAffectedArea)
);

/**
 * @swagger
 * /api/affectedArea/delete-affected-area/{id}:
 *   delete:
 *     summary: Delete a specific record of  affected area
 *     tags: [ affected area]
 */
affectedAreaRoutes.delete(
  "/delete-affected-area/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 450, "affected_area"),
  controllerWrapper(deleteAffectedArea)
);

export default affectedAreaRoutes;
