// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createUpdatedAppointmentType,
  deleteAppointmentType,
  viewAllAppointmentType,
} from "../controller/appointmentTypeController.js";
import {  validateRequest } from "../helper/general.js";
import { appointmentTypeValidationSchema } from "../validationsRecord/validationSchema.js";
import { controllerWrapper } from "../helper/wrapper.js";
/**
 * @swagger
 * tags:
 *   name: Appointment Type
 *   description: API operations for managing all appointment types
 */
const appointmentTypeRoutes = Router();

/**
 * @swagger
 * /api/appointmentType/create-update-appointment-type:
 *   post:
 *     summary: Create or update an appointment type
 *     tags: [Appointment Type]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               assignment:
 *                 type: string
 *               designated:
 *                 type: string
 *               applicable:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: { id: "id", name: "name", assignment: "assignment", designated: "Designated", applicable: "Applicable", organization: [] }
 */
appointmentTypeRoutes.post(
  "/create-update-appointment-type",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      169,
      "appointment_type"
    )(req, res, next);
  },
  validateRequest(appointmentTypeValidationSchema),
  controllerWrapper(createUpdatedAppointmentType)
);

/**
 * @swagger
 * /api/appointmentType/get-appointment-type:
 *   get:
 *     summary: Get all appointment types
 *     tags: [Appointment Type]
 */
appointmentTypeRoutes.get(
  "/get-appointment-type/",
  verifyToken,
  sideBarPermissionCheck("view", 169, "appointment_type"),
  controllerWrapper(viewAllAppointmentType)
);

/**
 * @swagger
 * /api/appointmentType/get-appointment-type/{id}:
 *   get:
 *     summary: Get a specific appointment type
 *     tags: [Appointment Type]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the appointment type to retrieve
 *         schema:
 *           type: string
 */
appointmentTypeRoutes.get(
  "/get-appointment-type/:id",
  verifyToken,
  sideBarPermissionCheck("view", 169, "appointment_type"),
  controllerWrapper(viewAllAppointmentType)
);

/**
 * @swagger
 * /api/appointmentType/delete-appointment-type/{id}:
 *   delete:
 *     summary: Delete a specific appointment type
 *     tags: [Appointment Type]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the appointment type to delete
 *         schema:
 *           type: string
 */
appointmentTypeRoutes.delete(
  "/delete-appointment-type/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 169, "appointment_type"),
  controllerWrapper(deleteAppointmentType)
);

export default appointmentTypeRoutes;
