// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateAssessmentLocation,
  deleteAssessmentLocation,
  viewAssessmentLocation,
} from "../controller/assessmentLocationController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Assessment Location
 *   description: API operations for managing all Assessment Location
 */
const assessmentLocationRoutes = Router();

/**
 * @swagger
 * /api/assessmentLocation/create-update-assessment-location:
 *   post:
 *     summary: Create or update an Assessment Location
 *     tags: [Assessment Location]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the assessment location"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Assessment Location created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
assessmentLocationRoutes.post(
  "/create-update-assessment-location",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      298,
      "assessment_location"
    )(req, res, next);
  },
  controllerWrapper(createUpdateAssessmentLocation)
);

/**
 * @swagger
 * /api/assessmentLocation/get-assessment-location:
 *   get:
 *     summary: Get list of all Assessment Locations
 *     tags: [Assessment Location]
 *     responses:
 *       200:
 *         description: List of assessment locations
 *       500:
 *         description: Internal server error
 */
assessmentLocationRoutes.get(
  "/get-assessment-location",
  verifyToken,
  sideBarPermissionCheck("view", 298, "assessment_location"),
  controllerWrapper(viewAssessmentLocation)
);

/**
 * @swagger
 * /api/assessmentLocation/get-assessment-location/{id}:
 *   get:
 *     summary: Get a specific record of Assessment Location
 *     tags: [Assessment Location]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the assessment location
 *     responses:
 *       200:
 *         description: Assessment Location data
 *       404:
 *         description: Assessment Location not found
 *       500:
 *         description: Internal server error
 */
assessmentLocationRoutes.get(
  "/get-assessment-location/:id",
  verifyToken,
  sideBarPermissionCheck("view", 298, "assessment_location"),
  controllerWrapper(viewAssessmentLocation)
);

/**
 * @swagger
 * /api/assessmentLocation/delete-assessment-location/{id}:
 *   delete:
 *     summary: Delete a specific record of Assessment Location
 *     tags: [Assessment Location]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the assessment location to delete
 *     responses:
 *       200:
 *         description: Assessment Location successfully deleted
 *       404:
 *         description: Assessment Location not found
 *       500:
 *         description: Internal server error
 */
assessmentLocationRoutes.delete(
  "/delete-assessment-location/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 298, "assessment_location"),
  controllerWrapper(deleteAssessmentLocation)
);

export default assessmentLocationRoutes;
