// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedAuditTemplate,
  deleteAuditTemplate,
  viewAllAuditTemplate,
} from "../controller/auditTemplateController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { auditTemplateCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Audit Template
 *   description: API operations for managing all Audit Template
 */
const auditTemplateRoutes = Router();

/**
 * @swagger
 * /api/auditTemplate/create-update-audit-template:
 *   post:
 *     summary: Create or update an Audit Template
 *     tags: [Audit Template]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               questions:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *               quiz:
 *                 type: string
 *           example: {
 *             id: "id",
 *             name: "name",
 *             description: "description",
 *             organization: [],
 *             questions: ["question1", "question2"],
 *             department: "department",
 *             quiz: "quiz"
 *           }
 */
auditTemplateRoutes.post(
  "/create-update-audit-template",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      145,
      "audit_template"
    )(req, res, next);
  },
  validateRequest(auditTemplateCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedAuditTemplate)
);

/**
 * @swagger
 * /api/auditTemplate/get-audit-template:
 *   get:
 *     summary: Get a list of all Audit Template
 *     tags: [Audit Template]
 */
auditTemplateRoutes.get(
  "/get-audit-template",
  verifyToken,
  sideBarPermissionCheck("view", 145, "audit_template"),
  controllerWrapper(viewAllAuditTemplate)
);

/**
 * @swagger
 * /api/auditTemplate/get-audit-template/{id}:
 *   get:
 *     summary: Get a specific record of Audit Template
 *     tags: [Audit Template]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit template to retrieve
 *         schema:
 *           type: string
 */
auditTemplateRoutes.get(
  "/get-audit-template/:id",
  verifyToken,
  sideBarPermissionCheck("view", 145, "audit_template"),
  controllerWrapper(viewAllAuditTemplate)
);

/**
 * @swagger
 * /api/auditTemplate/delete-audit-template/{id}:
 *   delete:
 *     summary: Delete a specific record of Audit Template
 *     tags: [Audit Template]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit template to delete
 *         schema:
 *           type: string
 */
auditTemplateRoutes.delete(
  "/delete-audit-template/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 145, "audit_template"),
  controllerWrapper(deleteAuditTemplate)
);

export default auditTemplateRoutes;
