// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateCertificates,
  deleteCertificates,
  viewCertificates,
} from "../controller/certificatesController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Certificates
 *   description: API operations for managing all Certificates
 */
const certificatesRoutes = Router();

/**
 * @swagger
 * /api/certificates/create-update-certificates:
 *   post:
 *     summary: Create or update an Certificates
 *     tags: [Certificates]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Certificate Name"
 *             description: "Description of the Certificates"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Certificates created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
certificatesRoutes.post(
  "/create-update-certificates",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      251,
      "certificates"
    )(req, res, next);
  },
  controllerWrapper(createUpdateCertificates)
);

/**
 * @swagger
 * /api/certificates/get-certificates:
 *   get:
 *     summary: Get list of all Certificates
 *     tags: [Certificates]
 *     responses:
 *       200:
 *         description: List of Certificates
 *       500:
 *         description: Internal server error
 */
certificatesRoutes.get(
  "/get-certificates",
  verifyToken,
  sideBarPermissionCheck("view", 251, "certificates"),
  controllerWrapper(viewCertificates)
);

/**
 * @swagger
 * /api/certificates/get-certificates/{id}:
 *   get:
 *     summary: Get a specific record of Certificates
 *     tags: [Certificates]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the Certificates
 *     responses:
 *       200:
 *         description: Certificates data
 *       404:
 *         description: Certificates not found
 *       500:
 *         description: Internal server error
 */
certificatesRoutes.get(
  "/get-certificates/:id",
  verifyToken,
  sideBarPermissionCheck("view", 251, "certificates"),
  controllerWrapper(viewCertificates)
);

/**
 * @swagger
 * /api/certificates/delete-certificates/{id}:
 *   delete:
 *     summary: Delete a specific record of Certificates
 *     tags: [Certificates]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the Certificates to delete
 *     responses:
 *       200:
 *         description: Certificates successfully deleted
 *       404:
 *         description: Certificates not found
 *       500:
 *         description: Internal server error
 */
certificatesRoutes.delete(
  "/delete-certificates/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 251, "certificates"),
  controllerWrapper(deleteCertificates)
);

export default certificatesRoutes;
