import { Router } from "express";
import {
  createUpdateContractorTrainingAttendance,
  getContractorTrainingAttendance,
  deleteContractorTrainingAttendance,
} from "../controller/contractorAttendanceRecordingController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const contractorAttendanceRecordingRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Contractor Training Attendance Recording
 *   description: API operations for managing training attendance recordings
 */

/**
 * @swagger
 * /api/contractorAttendanceRecording/create-update-contractor-attendance-recording:
 *   post:
 *     summary: Create or update an attendance recording
 *     tags: [Contractor Training Attendance Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorAttendanceRecording'
 *           example:
 *             organization: 1
 *             department: 2
 *             training_awareness_program: 3
 *             training_planned_start_date: "2024-07-17"
 *             training_actual_date: "2024-07-18"
 *             actual_training_service_provider: "Provider Name"
 *             training_venue: "Conference Room 1"
 *             training_total_cost: "1000"
 *             training_participation: "50 participants"
 *             document: "document_link"
 *     responses:
 *       200:
 *         description: The created or updated attendance recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorAttendanceRecording'
 */
contractorAttendanceRecordingRoutes.post(
  "/create-update-contractor-attendance-recording",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      294,
      "contractor_attendance_recording"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorTrainingAttendance)
);

/**
 * @swagger
 * /api/contractorAttendanceRecording/get-contractor-attendance-recording:
 *   get:
 *     summary: Get all attendance recordings
 *     tags: [Contractor Training Attendance Recording]
 *     responses:
 *       200:
 *         description: A list of attendance recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorAttendanceRecording'
 */
contractorAttendanceRecordingRoutes.get(
  "/get-contractor-attendance-recording",
  verifyToken,
  sideBarPermissionCheck("view", 294, "contractor_attendance_recording"),
  controllerWrapper(getContractorTrainingAttendance)
);

/**
 * @swagger
 * /api/contractorAttendanceRecording/get-contractor-attendance-recording/{id}:
 *   get:
 *     summary: Get an attendance recording by ID
 *     tags: [Contractor Training Attendance Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the attendance recording
 *     responses:
 *       200:
 *         description: The requested attendance recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorAttendanceRecording'
 */
contractorAttendanceRecordingRoutes.get(
  "/get-contractor-attendance-recording/:id",
  verifyToken,
  sideBarPermissionCheck("view", 294, "contractor_attendance_recording"),
  controllerWrapper(getContractorTrainingAttendance)
);

/**
 * @swagger
 * /api/contractorAttendanceRecording/delete-contractor-attendance-recording/{id}:
 *   delete:
 *     summary: Delete an attendance recording by ID
 *     tags: [Contractor Training Attendance Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the attendance recording
 *     responses:
 *       200:
 *         description: The attendance recording was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Attendance recording deleted successfully"
 */
contractorAttendanceRecordingRoutes.delete(
  "/delete-contractor-attendance-recording/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 294, "contractor_attendance_recording"),
  controllerWrapper(deleteContractorTrainingAttendance)
);

export default contractorAttendanceRecordingRoutes;
