// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateContractorForm,
  deleteContractorForm,
  viewContractorForm,
} from "../controller/contractorFormController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Forms
 *   description: API operations for managing all Contractor Forms
 */
const contractorFormRoutes = Router();

/**
 * @swagger
 * /api/contractorForm/create-update-contractor-form:
 *   post:
 *     summary: Create or update an Contractor Forms
 *     tags: [Contractor Forms]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the contractor form"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Contractor Form created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
contractorFormRoutes.post(
  "/create-update-contractor-form",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      327,
      "contractor_forms"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorForm)
);

/**
 * @swagger
 * /api/contractorForm/get-contractor-form:
 *   get:
 *     summary: Get list of all Contractor Forms
 *     tags: [Contractor Forms]
 *     responses:
 *       200:
 *         description: List of contractor forms
 *       500:
 *         description: Internal server error
 */
contractorFormRoutes.get(
  "/get-contractor-form",
  verifyToken,
  sideBarPermissionCheck("view", 327, "contractor_forms"),
  controllerWrapper(viewContractorForm)
);

/**
 * @swagger
 * /api/contractorForm/get-contractor-form/{id}:
 *   get:
 *     summary: Get a specific record of Contractor Forms
 *     tags: [Contractor Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the contractor form
 *     responses:
 *       200:
 *         description: Contractor Form data
 *       404:
 *         description: Contractor Form not found
 *       500:
 *         description: Internal server error
 */
contractorFormRoutes.get(
  "/get-contractor-form/:id",
  verifyToken,
  sideBarPermissionCheck("view", 327, "contractor_forms"),
  controllerWrapper(viewContractorForm)
);

/**
 * @swagger
 * /api/contractorForm/delete-contractor-form/{id}:
 *   delete:
 *     summary: Delete a specific record of Contractor Forms
 *     tags: [Contractor Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the contractor form to delete
 *     responses:
 *       200:
 *         description: Contractor Form successfully deleted
 *       404:
 *         description: Contractor Form not found
 *       500:
 *         description: Internal server error
 */
contractorFormRoutes.delete(
  "/delete-contractor-form/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 327, "contractor_forms"),
  controllerWrapper(deleteContractorForm)
);

export default contractorFormRoutes;
