// Import necessary modules
import { Router } from "express";
import {
  createUpdateContractorMeetingRecording,
  viewAllContractorMeetingRecording,
  deleteContractorMeetingRecording,
  viewAllContractorMeetingRecordingById,
  getContractorMeetingRecordActionItem,
  meetingNotesStatusChange,
  updateApprovalStatus,
} from "../controller/contractorMeetingRecordingController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { validateRequest } from "../helper/general.js";
import { meetingCreateUpdateRecordingSchema } from "../validationsRecord/validationSchema.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Meeting Recording
 *   description: API operations for managing all Contractor Meeting Recording
 */
const contractorMeetingRecordingRoutes = Router();

/**
 * @swagger
 * /api/contractorMeetingRecording/create-update-contractor-meeting-recording:
 *   post:
 *     summary: Create or update a Contractor Meeting Recording
 *     tags: [Contractor Meeting Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The auto-generated ID of the contractor meeting recording
 *               scheduled_meeting:
 *                 type: integer
 *                 description: ID of the scheduled meeting
 *               actual_meeting_date_from:
 *                 type: string
 *                 format: date-time
 *                 description: Start date and time of the actual meeting
 *               actual_meeting_date_to:
 *                 type: string
 *                 format: date-time
 *                 description: End date and time of the actual meeting
 *               meeting_duration:
 *                 type: string
 *                 description: Duration of the meeting
 *               meeting_scriber:
 *                 type: integer
 *                 description: ID of the person who scribes the meeting
 *               participants:
 *                 type: string
 *                 description: JSON array of participants
 *               meeting_notes:
 *                 type: string
 *                 description: Notes for the meeting
 *               status:
 *                 type: string
 *                 enum: [New, Pending, Approved, Rejected]
 *                 description: Status of the meeting
 *               comment:
 *                 type: string
 *                 description: Additional comments
 *               created_by:
 *                 type: integer
 *                 description: ID of the creator
 *               organization:
 *                 type: integer
 *                 description: Organization ID
 *               department:
 *                 type: integer
 *                 description: Department ID
 *               deleted:
 *                 type: integer
 *                 description: Deleted flag (0 or 1)
 *               created_at:
 *                 type: string
 *                 format: date-time
 *                 description: Creation timestamp
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *                 description: Last update timestamp
 *               updated_by:
 *                 type: integer
 *                 description: ID of the last updater
 *           example:
 *             id: 1
 *             scheduled_meeting: 123
 *             actual_meeting_date_from: "2024-01-01T10:00:00Z"
 *             actual_meeting_date_to: "2024-01-01T11:00:00Z"
 *             meeting_duration: "60"
 *             meeting_scriber: 456
 *             participants: "[\"Participant 1\", \"Participant 2\"]"
 *             meeting_notes: "Notes for the meeting"
 *             status: "New"
 *             comment: "Additional comments"
 *             created_by: 1
 *             organization: 3
 *             department: 2
 *             deleted: 0
 *             created_at: "2024-01-01T10:00:00Z"
 *             updated_at: "2024-01-01T11:00:00Z"
 *             updated_by: 1
 *     responses:
 *       200:
 *         description: The created or updated contractor meeting recording
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                   description: The ID of the contractor meeting recording
 *                 message:
 *                   type: string
 *                   description: Success message
 */

contractorMeetingRecordingRoutes.post(
  "/create-update-contractor-meeting-recording",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      287,
      "contractor_meeting_recording"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorMeetingRecording)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/get-contractor-meeting-recording:
 *   get:
 *     summary: Get all list of Contractor Meeting Recording
 *     tags: [Contractor Meeting Recording]
 *     responses:
 *       200:
 *         description: A list of all meeting recordings
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
contractorMeetingRecordingRoutes.get(
  "/get-contractor-meeting-recording",
  verifyToken,
  sideBarPermissionCheck("view", 287, "contractor_meeting_recording"),
  controllerWrapper(viewAllContractorMeetingRecording)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/get-contractor-meeting-recording/{id}:
 *   get:
 *     summary: Get a specific record of Contractor Meeting Recording
 *     tags: [Contractor Meeting Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting recording
 *     responses:
 *       200:
 *         description: A specific meeting recording record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
contractorMeetingRecordingRoutes.get(
  "/get-contractor-meeting-recording/:id",
  verifyToken,
  sideBarPermissionCheck("view", 287, "contractor_meeting_recording"),
  controllerWrapper(viewAllContractorMeetingRecording)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/get-contractor-meeting-recordingById/{meetingId}:
 *   get:
 *     summary: Get a specific record of Contractor Meeting Recording by meetingId
 *     tags: [Contractor Meeting Recording]
 *     parameters:
 *       - in: path
 *         name: meetingId
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting
 *     responses:
 *       200:
 *         description: A specific contractor Meeting Recording
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
contractorMeetingRecordingRoutes.get(
  "/get-contractor-meeting-recordingById/:meetingId",
  verifyToken,
  sideBarPermissionCheck("view", 287, "contractor_meeting_recording"),
  controllerWrapper(viewAllContractorMeetingRecordingById)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/delete-contractor-meeting-recording/{id}:
 *   delete:
 *     summary: Delete a specific record of Contractor Meeting Recording
 *     tags: [Contractor Meeting Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting recording
 *     responses:
 *       200:
 *         description: Successfully deleted
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
contractorMeetingRecordingRoutes.delete(
  "/delete-contractor-meeting-recording/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 287, "contractor_meeting_recording"),
  controllerWrapper(deleteContractorMeetingRecording)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/get-contractor-meeting-recording-action-item:
 *   get:
 *     summary: Get action items from Contractor Meeting Recording
 *     tags: [Contractor Meeting Recording]
 *     responses:
 *       200:
 *         description: A list of action items
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
contractorMeetingRecordingRoutes.get(
  "/get-contractor-meeting-recording-action-item",
  verifyToken,
  sideBarPermissionCheck("view", 287, "contractor_meeting_recording"),
  controllerWrapper(getContractorMeetingRecordActionItem)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/meeting-notes-status-change:
 *   post:
 *     summary: Change the status of Meeting Notes
 *     tags: [Contractor Meeting Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: {
 *             "id": 1,
 *             "meeting_action_progress": "In Progress",
 *             "status": "Updated"
 *           }
 */
contractorMeetingRecordingRoutes.post(
  "/meeting-notes-status-change",
  verifyToken,
  // sideBarPermissionCheck("create", 287, "contractor_meeting_recording"),
  controllerWrapper(meetingNotesStatusChange)
);

/**
 * @swagger
 * /api/contractorMeetingRecording/update-approval-status:
 *   get:
 *     summary: Update the approval status
 *     tags: [Contractor Meeting Recording]
 *     responses:
 *       200:
 *         description: Approval status updated successfully
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
contractorMeetingRecordingRoutes.get(
  "/update-approval-status",
  controllerWrapper(updateApprovalStatus)
);

export default contractorMeetingRecordingRoutes;
