import { Router } from "express";
import {
  createUpdateContractorNCR,
  getContractorNCR,
  deleteContractorNcrRecording,
  uploadSignatureContractorNCR,
} from "../controller/contractorNcrRecordingController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const contractorNcrRecordingRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Contractor Ncr Recording
 *   description: API operations for managing contractor meetings
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorNcrRecording:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated id of the NCR recording
 *         organization:
 *           type: integer
 *           description: Organization id
 *         department:
 *           type: integer
 *           description: Department id
 *         raised_by:
 *           type: string
 *           description: Name of the person who raised the NCR
 *         stop_certificate_id_no:
 *           type: string
 *           description: Stop certificate ID number
 *         ncr_id_no:
 *           type: string
 *           description: NCR ID number
 *         reasons:
 *           type: string
 *           description: Reasons for the NCR
 *         non_conformance_description:
 *           type: string
 *           description: Description of the non-conformance
 *         requirements_contravened:
 *           type: string
 *           description: Requirements that were contravened
 *         conditions_of_stop_certificate:
 *           type: string
 *           description: Conditions of the stop certificate
 *         recipient_full_name:
 *           type: string
 *           description: Full name of the recipient
 *         recipient_role_name:
 *           type: string
 *           description: Role name of the recipient
 *         recipient_contact_number:
 *           type: string
 *           description: Contact number of the recipient
 *         recipient_date_received:
 *           type: string
 *           format: date
 *           description: Date the recipient received the NCR
 *         recipient_employee_number:
 *           type: string
 *           description: Employee number of the recipient
 *         recipient_acceptance:
 *           type: string
 *           description: Acceptance status of the recipient
 *         issuer_full_name:
 *           type: string
 *           description: Full name of the issuer
 *         issuer_role_name:
 *           type: string
 *           description: Role name of the issuer
 *         issuer_contact_number:
 *           type: string
 *           description: Contact number of the issuer
 *         issuer_date_received:
 *           type: string
 *           format: date
 *           description: Date the issuer received the NCR
 *         issuer_time:
 *           type: string
 *           format: time
 *           description: Time the issuer received the NCR
 *         process_correction_verification:
 *           type: string
 *           description: Verification of process correction
 *         necessary_conditions_met:
 *           type: boolean
 *           description: Whether necessary conditions were met
 *         stop_certificate_to_be_lifted:
 *           type: boolean
 *           description: Whether the stop certificate is to be lifted
 *         verifier_full_name:
 *           type: string
 *           description: Full name of the verifier
 *         verifier_role_name:
 *           type: string
 *           description: Role name of the verifier
 *         verifier_contact_number:
 *           type: string
 *           description: Contact number of the verifier
 *         verifier_date:
 *           type: string
 *           format: date
 *           description: Date of verification
 *         date_lifted:
 *           type: string
 *           format: date
 *           description: Date the stop certificate was lifted
 *         time_lifted:
 *           type: string
 *           format: time
 *           description: Time the stop certificate was lifted
 *         compliance_to_mitigation:
 *           type: string
 *           description: Compliance to mitigation measures
 *         original_issuer_signature:
 *           type: string
 *           description: Signature of the original issuer
 *         lifting_acceptance_signature:
 *           type: string
 *           description: Signature for lifting acceptance
 *         deleted:
 *           type: integer
 *           description: Soft delete flag
 *         created_by:
 *           type: integer
 *           description: Id of the creator
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: Creation timestamp
 *         updated_by:
 *           type: integer
 *           description: Id of the updater
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: Update timestamp
 *       example:
 *         id: 1
 *         organization: 1
 *         department: 1
 *         raised_by: John Doe
 *         stop_certificate_id_no: SC123456
 *         ncr_id_no: NCR123456
 *         reasons: Some reasons
 *         non_conformance_description: Some description
 *         requirements_contravened: Some requirements
 *         conditions_of_stop_certificate: Some conditions
 *         recipient_full_name: Jane Doe
 *         recipient_role_name: Manager
 *         recipient_contact_number: "1234567890"
 *         recipient_date_received: "2024-07-22"
 *         recipient_employee_number: EMP123
 *         recipient_acceptance: Accepted
 *         issuer_full_name: John Smith
 *         issuer_role_name: Supervisor
 *         issuer_contact_number: "0987654321"
 *         issuer_date_received: "2024-07-22"
 *         issuer_time: "12:00:00"
 *         process_correction_verification: Verified
 *         necessary_conditions_met: true
 *         stop_certificate_to_be_lifted: true
 *         verifier_full_name: Jane Smith
 *         verifier_role_name: Inspector
 *         verifier_contact_number: "1122334455"
 *         verifier_date: "2024-07-23"
 *         date_lifted: "2024-07-24"
 *         time_lifted: "14:00:00"
 *         compliance_to_mitigation: Compliant
 *         original_issuer_signature: JohnSmithSignature
 *         lifting_acceptance_signature: JaneSmithSignature
 */

/**
 * @swagger
 * /api/contractorNcrRecording/create-update-contractor-ncr:
 *   post:
 *     summary: Create or update an NCR recording
 *     tags: [Contractor Ncr Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *       example:
 *         id: 1
 *         organization: 1
 *         department: 1
 *         raised_by: John Doe
 *         stop_certificate_id_no: SC123456
 *         ncr_id_no: NCR123456
 *         reasons: Some reasons
 *         non_conformance_description: Some description
 *         requirements_contravened: Some requirements
 *         conditions_of_stop_certificate: Some conditions
 *         recipient_full_name: Jane Doe
 *         recipient_role_name: Manager
 *         recipient_contact_number: "1234567890"
 *         recipient_date_received: "2024-07-22"
 *         recipient_employee_number: EMP123
 *         recipient_acceptance: Accepted
 *         issuer_full_name: John Smith
 *         issuer_role_name: Supervisor
 *         issuer_contact_number: "0987654321"
 *         issuer_date_received: "2024-07-22"
 *         issuer_time: "12:00:00"
 *         process_correction_verification: Verified
 *         necessary_conditions_met: true
 *         stop_certificate_to_be_lifted: true
 *         verifier_full_name: Jane Smith
 *         verifier_role_name: Inspector
 *         verifier_contact_number: "1122334455"
 *         verifier_date: "2024-07-23"
 *         date_lifted: "2024-07-24"
 *         time_lifted: "14:00:00"
 *         compliance_to_mitigation: Compliant
 *         original_issuer_signature: JohnSmithSignature
 *         lifting_acceptance_signature: JaneSmithSignature
 *     responses:
 *       200:
 *         description: The created/updated NCR recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorNcrRecording'
 */
contractorNcrRecordingRoutes.post(
  "/create-update-contractor-ncr",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      291,
      "contractor_ncr_recording"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorNCR)
);

/**
 * @swagger
 * /api/contractorNcrRecording/get-contractor-ncr:
 *   get:
 *     summary: Get all NCR recordings
 *     tags: [Contractor Ncr Recording]
 *     responses:
 *       200:
 *         description: List of all NCR recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorNcrRecording'
 */
contractorNcrRecordingRoutes.get(
  "/get-contractor-ncr",
  verifyToken,
  sideBarPermissionCheck("view", 291, "contractor_ncr_recording"),
  controllerWrapper(getContractorNCR)
);

/**
 * @swagger
 * /api/contractorNcrRecording/get-contractor-ncr/{id}:
 *   get:
 *     summary: Get an NCR recording by id
 *     tags: [Contractor Ncr Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The NCR recording id
 *     responses:
 *       200:
 *         description: The NCR recording by id
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorNcrRecording'
 */
contractorNcrRecordingRoutes.get(
  "/get-contractor-ncr/:id",
  verifyToken,
  sideBarPermissionCheck("view", 291, "contractor_ncr_recording"),
  controllerWrapper(getContractorNCR)
);

/**
 * @swagger
 * /api/contractorNcrRecording/delete-contractor-ncr/{id}:
 *   delete:
 *     summary: Delete an NCR recording by id
 *     tags: [Contractor Ncr Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The NCR recording id
 *     responses:
 *       200:
 *         description: NCR recording deleted
 */
contractorNcrRecordingRoutes.delete(
  "/delete-contractor-ncr/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 291, "contractor_ncr_recording"),
  controllerWrapper(deleteContractorNcrRecording)
);

/**
 * @swagger
 * /api/contractorNcrRecording/upload-ncr-recording-signature:
 *   post:
 *     summary: upload document of contractor ncr
 *     tags: [Contractor Ncr Recording]
 */
contractorNcrRecordingRoutes.post(
  "/upload-ncr-recording-signature",
  verifyToken,
  controllerWrapper(uploadSignatureContractorNCR)
);

export default contractorNcrRecordingRoutes;
