import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";

import {
  createUpdateContractorTrainingSchedule,
  getContractorTrainingSchedule,
  deleteContractorTrainingSchedule,
  fetchContractorTrainingScheduleData,
} from "../controller/contractorTrainingScheduleController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const contractorTrainingScheduleRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Contractor Training Schedule
 *   description: API operations for managing contractor training schedules
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorTrainingSchedule:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor training schedule
 *         training_program:
 *           type: integer
 *           description: ID of the training program
 *         employees:
 *           type: string
 *           description: List of employees participating in the training
 *         training_provider:
 *           type: string
 *           enum: ["Internal", "External"]
 *           description: Type of training provider
 *         external_provider:
 *           type: string
 *           description: Name of the external provider
 *         provider_department:
 *           type: string
 *           description: Department of the provider
 *         period:
 *           type: string
 *           description: Period of the contractor training schedule
 *         financial_year:
 *           type: string
 *           description: Financial year of the training
 *         schedule_type:
 *           type: integer
 *           description: Type of schedule (e.g., single day, multiple days)
 *         single_day:
 *           type: string
 *           format: date
 *           description: Date for a single-day training
 *         single_start_time:
 *           type: string
 *           format: time
 *           description: Start time for a single-day training
 *         single_end_time:
 *           type: string
 *           format: time
 *           description: End time for a single-day training
 *         multiple_day:
 *           type: string
 *           format: date
 *           description: Start date for multiple-day training
 *         multiple_end_day:
 *           type: string
 *           format: date
 *           description: End date for multiple-day training
 *         calendar_start_date:
 *           type: string
 *           format: date
 *           description: Start date for calendar-based training
 *         calendar_end_date:
 *           type: string
 *           format: date
 *           description: End date for calendar-based training
 *         schedule_time:
 *           type: string
 *           description: Schedule time details for the training
 *         department:
 *           type: integer
 *           description: Department ID
 *         organization:
 *           type: integer
 *           description: Organization ID
 *         deleted:
 *           type: integer
 *           description: Deletion status of the record
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The creation date of the contractor training schedule
 *         created_by:
 *           type: integer
 *           description: ID of the creator
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The last update date of the contractor training schedule
 *         updated_by:
 *           type: integer
 *           description: ID of the last updater
 */

/**
 * @swagger
 * /api/contractorTrainingSchedule/create-update-contractor-training-schedule:
 *   post:
 *     summary: Create or update a contractor training schedule
 *     tags: [Contractor Training Schedule]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorTrainingSchedule'
 *           example:
 *             organization: 3
 *             employees: "[12, 10]" # Updated to reflect that it's a string representation of the list
 *             training_program: 45
 *             training_provider: "Internal"
 *             external_provider: ""
 *             provider_department: "HR"
 *             period: "Monthly"
 *             financial_year: "2024"
 *             schedule_type: 1
 *             single_day: ""
 *             single_start_time: ""
 *             single_end_time: ""
 *             multiple_day: "2024-07-23"
 *             multiple_end_day: "2024-07-24"
 *             calendar_start_date: "2024-07-01"
 *             calendar_end_date: "2024-07-31"
 *             schedule_time: "Full Day"
 *             department: 2
 *             deleted: 0
 *             created_by: 1
 *             updated_by: 1
 *     responses:
 *       200:
 *         description: The created or updated contractor training schedule
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorTrainingSchedule'
 */

contractorTrainingScheduleRoutes.post(
  "/create-update-contractor-training-schedule",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      264,
      "contractor_training_schedule"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorTrainingSchedule)
);

/**
 * @swagger
 * /api/contractorTrainingSchedule/get-contractor-training-schedule:
 *   get:
 *     summary: Get all contractor training schedules
 *     tags: [Contractor Training Schedule]
 *     responses:
 *       200:
 *         description: A list of contractor training schedules
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorTrainingSchedule'
 */
contractorTrainingScheduleRoutes.get(
  "/get-contractor-training-schedule",
  verifyToken,
  sideBarPermissionCheck("view", 264, "contractor_training_schedule"),
  controllerWrapper(getContractorTrainingSchedule)
);

/**
 * @swagger
 * /api/contractorTrainingSchedule/get-contractor-training-schedule/{id}:
 *   get:
 *     summary: Get a contractor training schedule by ID
 *     tags: [Contractor Training Schedule]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the contractor training schedule
 *     responses:
 *       200:
 *         description: The requested contractor training schedule
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorTrainingSchedule'
 */
contractorTrainingScheduleRoutes.get(
  "/get-contractor-training-schedule/:id",
  verifyToken,
  sideBarPermissionCheck("view", 264, "contractor_training_schedule"),
  controllerWrapper(getContractorTrainingSchedule)
);

/**
 * @swagger
 * /api/contractorTrainingSchedule/delete-contractor-training-schedule/{id}:
 *   delete:
 *     summary: Delete a contractor training schedule by ID
 *     tags: [Contractor Training Schedule]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the contractor training schedule
 *     responses:
 *       200:
 *         description: The contractor training schedule was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Contractor Training schedule deleted successfully"
 */
contractorTrainingScheduleRoutes.delete(
  "/delete-contractor-training-schedule/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 264, "contractor_training_schedule"),
  controllerWrapper(deleteContractorTrainingSchedule)
);

contractorTrainingScheduleRoutes.get(
  "/get-contractor-training-schedule",
  verifyToken,
  sideBarPermissionCheck("view", 264, "contractor_training_schedule"),
  fetchContractorTrainingScheduleData
);
export default contractorTrainingScheduleRoutes;
