// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedDocumentRepository,
  deleteDocumentRepository,
  viewAllDocumentRepository,
} from "../controller/documentRepositoryController.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Document Repository
 *   description: API operations for managing all Document Repositories
 */
const documentRepositoryRoutes = Router();

/**
 * @swagger
 * /api/documentRepository/create-update-document-repository:
 *   post:
 *     summary: Create or update a Document Repository
 *     tags: [Document Repository]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               document_type:
 *                 type: string
 *                 enum: [External, Internal]
 *               associated_audit:
 *                 type: string
 *               document_name:
 *                 type: string
 *               document_description:
 *                 type: string
 *               allowed_access_to:
 *                 type: string
 *               audit_execution_id:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *           example: {
 *             id: "id",
 *             document_type: "External",
 *             associated_audit: "scheduled_audit_id",
 *             document_name: "document_name",
 *             document_description: "document_description",
 *             allowed_access_to: "allowed_access_to",
 *             audit_execution_id: "audit_execution_id",
 *             organization: [],
 *             department: "department"
 *           }
 */
documentRepositoryRoutes.post(
  "/create-update-document-repository",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      158,
      "document_repository"
    )(req, res, next);
  },
  // validateRequest(documentRepositoryCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedDocumentRepository)
);

/**
 * @swagger
 * /api/documentRepository/get-document-repository:
 *   get:
 *     summary: Get a list of all Document Repositories
 *     tags: [Document Repository]
 */
documentRepositoryRoutes.get(
  "/get-document-repository",
  verifyToken,
  sideBarPermissionCheck("view", 158, "document_repository"),
  controllerWrapper(viewAllDocumentRepository)
);

/**
 * @swagger
 * /api/documentRepository/get-document-repository/{id}:
 *   get:
 *     summary: Get a specific record of Document Repository
 *     tags: [Document Repository]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the Document Repository to retrieve
 *         schema:
 *           type: string
 */
documentRepositoryRoutes.get(
  "/get-document-repository/:id",
  verifyToken,
  sideBarPermissionCheck("view", 158, "document_repository"),
  controllerWrapper(viewAllDocumentRepository)
);

/**
 * @swagger
 * /api/documentRepository/delete-document-repository/{id}:
 *   delete:
 *     summary: Delete a specific record of Document Repository
 *     tags: [Document Repository]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the Document Repository to delete
 *         schema:
 *           type: string
 */
documentRepositoryRoutes.delete(
  "/delete-document-repository/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 158, "document_repository"),
  controllerWrapper(deleteDocumentRepository)
);

export default documentRepositoryRoutes;
