import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateHseAppointment,
  getHseAppointment,
  deleteHseAppointment,
  getHseAppointmentChart,
  uploadSignature,
} from "../controller/hseAppointmentController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { hseAppointmentCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

const hseAppointmentRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: HSE Appointment
 *   description: API operations for managing all HSE appointments
 */

/**
 * @swagger
 * /api/hseAppointment/create-update-hseAppointment:
 *   post:
 *     summary: Create or update an HSE appointment
 *     tags: [HSE Appointment]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               appoints_user_id:
 *                 type: integer
 *               legal_appointment_role:
 *                 type: string
 *               appointed_user_id:
 *                 type: integer
 *               name_of_appointment:
 *                 type: string
 *               effective_date:
 *                 type: string
 *                 format: date
 *               end_date:
 *                 type: string
 *                 format: date
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *           example:
 *             id: 1
 *             appoints_user_id: 123
 *             legal_appointment_role: "Manager"
 *             appointed_user_id: 456
 *             name_of_appointment: "Safety Inspection"
 *             effective_date: "2023-01-01"
 *             end_date: "2023-12-31"
 *             organization: 1
 *             department: 2
 */
hseAppointmentRoutes.post(
  "/create-update-hseAppointment",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      96,
      "hse_appointment"
    )(req, res, next);
  },
  validateRequest(hseAppointmentCreateUpdateValidationSchema),
  controllerWrapper(createUpdateHseAppointment)
);

/**
 * @swagger
 * /api/hseAppointment/get-hseAppointment:
 *   get:
 *     summary: Get all HSE appointments
 *     tags: [HSE Appointment]
 */
hseAppointmentRoutes.get(
  "/get-hseAppointment",
  verifyToken,
  sideBarPermissionCheck("view", 96, "hse_appointment"),
  controllerWrapper(getHseAppointment)
);

/**
 * @swagger
 * /api/hseAppointment/get-hseAppointment/{id}:
 *   get:
 *     summary: Get a specific HSE appointment record
 *     tags: [HSE Appointment]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the HSE appointment to retrieve
 */
hseAppointmentRoutes.get(
  "/get-hseAppointment/:id",
  verifyToken,
  sideBarPermissionCheck("view", 96, "hse_appointment"),
  controllerWrapper(getHseAppointment)
);

/**
 * @swagger
 * /api/hseAppointment/delete-hseAppointment/{id}:
 *   delete:
 *     summary: Delete a specific HSE appointment record
 *     tags: [HSE Appointment]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the HSE appointment to delete
 */
hseAppointmentRoutes.delete(
  "/delete-hseAppointment/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 96, "hse_appointment"),
  controllerWrapper(deleteHseAppointment)
);

/**
 * @swagger
 * /api/hseAppointment/get-hse-appointment-chart:
 *   get:
 *     summary: Get chart for HSE employee
 *     tags: [HSE Appointment]
 */
hseAppointmentRoutes.get(
  "/get-hse-appointment-chart",
  verifyToken,
  sideBarPermissionCheck("view", 96, "hse_appointment"),
  controllerWrapper(getHseAppointmentChart)
);



/**
 * @swagger
 * /api/hseAppointment/upload-hse-appointment-signature:
 *   post:
 *     summary: Get chart for HSE employee
 *     tags: [HSE Appointment]
 */
hseAppointmentRoutes.post(
  "/upload-hse-appointment-signature",
  verifyToken,
  controllerWrapper(uploadSignature)
);





export default hseAppointmentRoutes;
