import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateIncidentPpe,
  deleteIncidentPpe,
  getParentIncidentPpeList,
  viewIncidentPpe,
} from "../controller/incidentPpeController.js";

/**
 * @swagger
 * tags:
 *   name:  incident ppe
 *   description: API operations for managing  incident ppe
 */
const incidentPpeRoutes = Router();

/**
 * @swagger
 * /api/incidentPpe/create-update-incident-ppe:
 *   post:
 *     summary: Create or update a  incident ppe
 *     tags: [ incident ppe]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
incidentPpeRoutes.post(
  "/create-update-incident-ppe",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 459, "incident_ppe")(req, res, next);
  },
  controllerWrapper(createUpdateIncidentPpe)
);

/**
 * @swagger
 * /api/incidentPpe/get-incident-ppe:
 *   get:
 *     summary: Get all records of  incident ppe
 *     tags: [ incident ppe]
 */
incidentPpeRoutes.get(
  "/get-incident-ppe",
  verifyToken,
  sideBarPermissionCheck("view", 459, "incident_ppe"),
  controllerWrapper(viewIncidentPpe)
);

/**
 * @swagger
 * /api/incidentPpe/get-incident-ppe/{id}:
 *   get:
 *     summary: Get a specific record of  incident ppe
 *     tags: [ incident ppe]
 */
incidentPpeRoutes.get(
  "/get-incident-ppe/:id",
  verifyToken,
  sideBarPermissionCheck("view", 459, "incident_ppe"),
  controllerWrapper(viewIncidentPpe)
);

/**
 * @swagger
 * /api/incidentPpe/get-incident-ppe-list:
 *   get:
 *     summary: Get the list for selecting parent category
 *     tags: [ incident ppe]
 *     parameters:
 *       - in: query
 *         organization : [1,2,34,4]
 *         schema:
 *           type: Array
 *         description: Array of organization to fetch the list
 */

incidentPpeRoutes.get("/get-incident-ppe-list", verifyToken, controllerWrapper(getParentIncidentPpeList));

/**
 * @swagger
 * /api/incidentPpe/delete-incident-ppe/{id}:
 *   delete:
 *     summary: Delete a specific record of  incident ppe
 *     tags: [ incident ppe]
 */
incidentPpeRoutes.delete(
  "/delete-incident-ppe/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 459, "incident_ppe"),
  controllerWrapper(deleteIncidentPpe)
);

export default incidentPpeRoutes;
