import { Router } from "express";
import {
  createUpdateKnowledgeManagementRecording,
  getKnowledgeManagementRecording,
  deleteKnowledgeManagementRecording,
  getKnowledgeManagementRecordingApprovalWorkflow,
  createComment,
} from "../controller/knowledgeManagementRecordingController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import KnowledgeComment from "../sequelize/knowledgeCommentSchema.js"; // for sync purpose only not to be removed

const knowledgeManagementRecordingRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Knowledge Management Recording
 *   description: API operations for managing knowledge management recordings
 */

/**
 * @swagger
 * /api/knowledgeManagementRecording/create-update-knowledge-management-recording:
 *   post:
 *     summary: Create or update a knowledge management recording
 *     tags: [Knowledge Management Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/KnowledgeManagementRecording'
 *           example:
 *             organization: 1
 *             department: 2
 *             knowledge_management_type: 1
 *             lesson_best_practice_id: "LBP123"
 *             title: "Best Practices in Project Management"
 *             category: 3
 *             description: "Detailed description of the lesson or best practice."
 *             author: 4
 *             date_of_submission: "2024-07-17"
 *             relevant_department: 5
 *             supporting_document: "document_link"
 *             recommendations: "Recommendations for future projects."
 *             key_insights: "Key insights from the project."
 *             keywords: "project, management, best practices"
 *             implementation_steps: "Step 1, Step 2, Step 3"
 *             approval_status: "Approved"
 *             review_date: "2024-08-01"
 *             comments: "Additional comments."
 *     responses:
 *       200:
 *         description: The created or updated knowledge management recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/KnowledgeManagementRecording'
 */
knowledgeManagementRecordingRoutes.post(
  "/create-update-knowledge-management-recording",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      218,
      "knowledge_management_recording"
    )(req, res, next);
  },
  controllerWrapper(createUpdateKnowledgeManagementRecording)
);

/**
 * @swagger
 * /api/knowledgeManagementRecording/get-knowledge-management-recording:
 *   get:
 *     summary: Get all knowledge management recordings
 *     tags: [Knowledge Management Recording]
 *     responses:
 *       200:
 *         description: A list of knowledge management recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/KnowledgeManagementRecording'
 */
knowledgeManagementRecordingRoutes.get(
  "/get-knowledge-management-recording",
  verifyToken,
  sideBarPermissionCheck("view", 218, "knowledge_management_recording"),
  controllerWrapper(getKnowledgeManagementRecording)
);

/**
 * @swagger
 * /api/knowledgeManagementRecording/get-knowledge-management-recording/{id}:
 *   get:
 *     summary: Get a knowledge management recording by ID
 *     tags: [Knowledge Management Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the knowledge management recording
 *     responses:
 *       200:
 *         description: The requested knowledge management recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/KnowledgeManagementRecording'
 */
knowledgeManagementRecordingRoutes.get(
  "/get-knowledge-management-recording/:id",
  verifyToken,
  sideBarPermissionCheck("view", 218, "knowledge_management_recording"),
  controllerWrapper(getKnowledgeManagementRecording)
);

/**
 * @swagger
 * /api/knowledgeManagementRecording/delete-knowledge-management-recording/{id}:
 *   delete:
 *     summary: Delete a knowledge management recording by ID
 *     tags: [Knowledge Management Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the knowledge management recording
 *     responses:
 *       200:
 *         description: The knowledge management recording was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Knowledge management recording deleted successfully"
 */
knowledgeManagementRecordingRoutes.delete(
  "/delete-knowledge-management-recording/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 218, "knowledge_management_recording"),
  controllerWrapper(deleteKnowledgeManagementRecording)
);



/**
 * @swagger
 * /api/knowledgeManagementRecording/get-knowledge-management-recording-approval-workflow:
 *   get:
 *     summary: Get all knowledge management recordings For Approval Workflow
 *     tags: [Knowledge Management Recording]
 *     responses:
 *       200:
 *         description: A list of knowledge management recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/KnowledgeManagementRecording'
 */
knowledgeManagementRecordingRoutes.get(
  "/get-knowledge-management-recording-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 218, "knowledge_management_recording"),
  controllerWrapper(getKnowledgeManagementRecordingApprovalWorkflow)
);






/**
 * @swagger
 * /api/knowledgeManagementRecording/get-knowledge-management-recording-approval-workflow/{id}:
 *   get:
 *     summary: Get all knowledge management recordings For Approval Workflow by id
 *     tags: [Knowledge Management Recording]
 *     responses:
 *       200:
 *         description: A list of knowledge management recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/KnowledgeManagementRecording'
 */
knowledgeManagementRecordingRoutes.get(
  "/get-knowledge-management-recording-approval-workflow/:id",
  verifyToken,
  sideBarPermissionCheck("view", 218, "knowledge_management_recording"),
  controllerWrapper(getKnowledgeManagementRecordingApprovalWorkflow)
);


/**
 * @swagger
 * /api/knowledgeManagementRecording/knowledge-comment:
 *   post:
 *     summary: Comment on Knowledge Management Recording
 *     tags: [Knowledge Management Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               knowledge_recording_id:
 *                 type: string
 *               comments:
 *                 type: string
 *           example:
 *             knowledge_recording_id: "1"
 *             comments: "This is a comment."
 */
knowledgeManagementRecordingRoutes.post(
  "/knowledge-comment",
  verifyToken,
  sideBarPermissionCheck("view", 218, "knowledge_management_recording"),
  controllerWrapper(createComment)
);




export default knowledgeManagementRecordingRoutes;

