// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateLegislation,
  deleteLegislation,
  viewLegislation,
} from "../controller/legislationController.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { legislationValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Legislation
 *   description: API operations for managing all legislation
 */
const legislationRoutes = Router();

/**
 * @swagger
 * /api/legislation/create-update-legislation:
 *   post:
 *     summary: Create or Update a Legislation
 *     tags: [Legislation]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             she_legislation: "Legislation Name"
 *             organization: 1
 *             department: 2
 *             document_title: "Document Title"
 *     responses:
 *       200:
 *         description: Legislation created or updated successfully
 *       400:
 *         description: Bad request
 *       500:
 *         description: Internal server error
 */
legislationRoutes.post(
  "/create-update-legislation",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      152,
      "legislation"
    )(req, res, next);
  },
  validateRequest(legislationValidationSchema),
  controllerWrapper(createUpdateLegislation)
);

/**
 * @swagger
 * /api/legislation/get-legislation:
 *   get:
 *     summary: Get All Legislation List
 *     tags: [Legislation]
 *     responses:
 *       200:
 *         description: A list of legislation
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
legislationRoutes.get(
  "/get-legislation",
  verifyToken,
  sideBarPermissionCheck("view", 152, "legislation_doc"),
  controllerWrapper(viewLegislation)
);

/**
 * @swagger
 * /api/legislation/get-legislation/{id}:
 *   get:
 *     summary: Get A Specific Legislation Record
 *     tags: [Legislation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the legislation
 *     responses:
 *       200:
 *         description: A specific legislation record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
legislationRoutes.get(
  "/get-legislation/:id",
  verifyToken,
  sideBarPermissionCheck("view", 152, "legislation_doc"),
  controllerWrapper(viewLegislation)
);

/**
 * @swagger
 * /api/legislation/delete-legislation/{id}:
 *   delete:
 *     summary: Delete A Specific Legislation Record
 *     tags: [Legislation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the legislation to delete
 *     responses:
 *       200:
 *         description: Legislation deleted successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
legislationRoutes.delete(
  "/delete-legislation/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 152, "legislation_doc"),
  controllerWrapper(deleteLegislation)
);

export default legislationRoutes;
