// Import necessary modules
import { Router } from "express";
import { publicUrlAccess, verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deletePolicy,
  policyComment,
  policyCreateUpdate,
  updatePolicyApprovalWorkflow,
  viewPolicy,
  viewPolicyApprovalWorkFlow,
  viewPublicPolicy,
} from "../controller/policiesController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Policy
 *   description: API Operations For Managing Policy
 */
const PolicyRoutes = Router();
/**
 * @swagger
 * /api/policy/create-update-policy:
 *   post:
 *     summary: Create or update a Policy
 *     tags: [Policy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               organization:
 *                 type: integer
 *               new_sop:
 *                 type: string
 *               policy_title:
 *                 type: string
 *               document_classification:
 *                 type: string
 *               effective_date:
 *                 type: string
 *                 format: date
 *               next_review_date:
 *                 type: string
 *                 format: date
 *               policy_status:
 *                 type: string
 *               policy_author:
 *                 type: integer
 *               policy_reviewer:
 *                 type: array
 *                 items:
 *                   type: integer
 *               policy_owner:
 *                 type: integer
 *               policy_approver:
 *                 type: integer
 *               policy_purpose:
 *                 type: string
 *               policy_scope:
 *                 type: string
 *               policy_terms:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     term:
 *                       type: string
 *                     description:
 *                       type: string
 *               policy_abbreviations:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     abbreviation:
 *                       type: string
 *                     description:
 *                       type: string
 *               roles_responsibilities:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     role_name:
 *                       type: integer
 *                     responsibility:
 *                       type: string
 *               policy_statement:
 *                 type: string
 *               policy_description:
 *                 type: string
 *               document_name:
 *                 type: string
 *               document_reference_number:
 *                 type: string
 *               version_number:
 *                 type: string
 *               revision_effective_date:
 *                 type: string
 *                 format: date
 *               details_of_addition:
 *                 type: string
 *               send_to_employee:
 *                 type: array
 *                 items:
 *                   type: integer
 *           example:
 *             organization: 14
 *             new_sop: "1"
 *             policy_title: "new policy31"
 *             document_classification: "Unclassified"
 *             effective_date: "2024-10-16"
 *             next_review_date: "2024-10-11"
 *             policy_status: "Complete"
 *             policy_author: 34
 *             policy_reviewer: [34, 33]
 *             policy_owner: 34
 *             policy_approver: 34
 *             policy_purpose: "<p>q</p>"
 *             policy_scope: "<p>q</p>"
 *             policy_terms:
 *               - term: "q"
 *                 description: "<p>q</p>"
 *             policy_abbreviations:
 *               - abbreviation: "q"
 *                 description: "<p>q</p>"
 *             roles_responsibilities:
 *               - role_name: 16
 *                 responsibility: "<p>q</p>"
 *             policy_statement: "<p>q</p>"
 *             policy_description: "<p>q</p>"
 *             document_name: "q"
 *             document_reference_number: "q"
 *             version_number: ""
 *             revision_effective_date: "2024-10-28"
 *             details_of_addition: "<p>q</p>"
 *             department: "department"
 */

PolicyRoutes.post(
  "/create-update-policy",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 69, "policy")(
      req,
      res,
      next
    );
  },
  controllerWrapper(policyCreateUpdate)
);

/**
 * @swagger
 * /api/policy/get-policy:
 *   get:
 *     summary: Get list of all Policy
 *     tags: [Policy]
 */
PolicyRoutes.get(
  "/get-policy",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewPolicy)
);

/**
 * @swagger
 * /api/policy/get-policy/{id}:
 *   get:
 *     summary: Get A Specific Policy record
 *     tags: [Policy]
 */
PolicyRoutes.get(
  "/get-policy/:id",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewPolicy)
);

/**
 * @swagger
 * /api/policy/delete-policy/{id}:
 *   delete:
 *     summary: Delete a specific record of Policy
 *     tags: [Policy]
 */
PolicyRoutes.delete(
  "/delete-policy/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 69, "policy"),
  controllerWrapper(deletePolicy)
);

/**
 * @swagger
 * /api/policy/policy-comment:
 *   post:
 *     summary: Comment on Policy
 *     tags: [Policy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               policy_id:
 *                 type: string
 *               comments:
 *                 type: string
 *           example:
 *             policy_id: "1"
 *             comments: "This is a comment."
 */
PolicyRoutes.post(
  "/policy-comment",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(policyComment)
);

/**
 * @swagger
 * /api/policy/get-policy-approval-workflow:
 *   get:
 *     summary: Get all Policies for Approval Workflow flow
 *     tags: [Policy]
 */
PolicyRoutes.get(
  "/get-policy-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewPolicyApprovalWorkFlow)
);



/**
 * @swagger
 * /api/policy/get-policy-public-url/{id}:
 *   get:
 *     summary: Get public URL for a specific Policy
 *     tags: [Policy]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Policy
 */
PolicyRoutes.get(
  "/get-policy-public-url/:id",
  verifyToken,
  publicUrlAccess,
  controllerWrapper(viewPublicPolicy)
);



/**
 * @swagger
 * /api/policy/update-policy-status/{id}:
 *   post:
 *     summary: update policy status
 *     tags: [Policy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               policy_approver:
 *                 type: string
 *           example:
 *             policy_approver: "policy_approver"
 */
PolicyRoutes.post(
  "/update-policy-approval-workflow/:id",
  verifyToken,
  controllerWrapper(updatePolicyApprovalWorkflow)
);






export default PolicyRoutes;
