// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  deleteSheLegalRegister,
  sheLegalSearchBarFilter,
  storeSheLegalWithApi,
  updateSheLegalRegister,
  viewSheLegalRegister,
} from "../controller/sheLegalRegisterController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: She Legal Register
 *   description: API operations for managing all She Legal Register
 */
const sheLegalRegisterRoutes = Router();

/**
 * @swagger
 * /api/sheLegalRegister/store-legal-record:
 *   get:
 *     summary: Store legal register record in database
 *     tags: [She Legal Register]
 *     responses:
 *       200:
 *         description: Successfully stored legal register record
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *             example:
 *               message: "Legal register record successfully stored."
 */
sheLegalRegisterRoutes.get("/store-legal-record", controllerWrapper(storeSheLegalWithApi));

/**
 * @swagger
 * /api/sheLegalRegister/get-legal-record:
 *   get:
 *     summary: Get a list of all legal register records
 *     tags: [She Legal Register]
 *     responses:
 *       200:
 *         description: A list of legal register records
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   title:
 *                     type: string
 *                   regulatory_body:
 *                     type: string
 *                   legislation_sector_type:
 *                     type: string
 *             example:
 *               - id: 1
 *                 title: "Environmental Protection Act"
 *                 regulatory_body: "Department of Environment"
 *                 legislation_sector_type: "Environmental"
 */
sheLegalRegisterRoutes.get(
  "/get-legal-record",
  verifyToken,
  controllerWrapper(viewSheLegalRegister)
);

/**
 * @swagger
 * /api/sheLegalRegister/get-legal-search-record:
 *   get:
 *     summary: Get a list of filtered legal search records
 *     tags: [She Legal Register]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               search_term:
 *                 type: string
 *             example:
 *               search_term: "Environmental"
 *     responses:
 *       200:
 *         description: A list of filtered legal search records
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   title:
 *                     type: string
 *                   regulatory_body:
 *                     type: string
 *             example:
 *               - id: 1
 *                 title: "Environmental Protection Act"
 *                 regulatory_body: "Department of Environment"
 */
sheLegalRegisterRoutes.get(
  "/get-legal-search-record",
  verifyToken,
  controllerWrapper(sheLegalSearchBarFilter)
);

/**
 * @swagger
 * /api/sheLegalRegister/get-legal-record/{id}:
 *   get:
 *     summary: Get specific legal register record
 *     tags: [She Legal Register]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the legal register record
 *     responses:
 *       200:
 *         description: Legal register record details
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 title:
 *                   type: string
 *                 regulatory_body:
 *                   type: string
 *                 legislation_sector_type:
 *                   type: string
 *             example:
 *               id: 1
 *               title: "Environmental Protection Act"
 *               regulatory_body: "Department of Environment"
 *               legislation_sector_type: "Environmental"
 */
sheLegalRegisterRoutes.get(
  "/get-legal-record/:id",
  verifyToken,
  controllerWrapper(viewSheLegalRegister)
);

/**
 * @swagger
 * /api/sheLegalRegister/delete-legal-record/{id}:
 *   delete:
 *     summary: Delete a specific record of She Legal Register
 *     tags: [She Legal Register]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the legal register record to delete
 *     responses:
 *       200:
 *         description: Successfully deleted the legal register record
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *             example:
 *               message: "Legal register record successfully deleted."
 */
sheLegalRegisterRoutes.delete(
  "/delete-legal-record/:id",
  verifyToken,
  controllerWrapper(deleteSheLegalRegister)
);

/**
 * @swagger
 * /api/sheLegalRegister/update-she-legal-register:
 *   post:
 *     summary: Create or update a She Legal Register record
 *     tags: [She Legal Register]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               risk_given_id:
 *                 type: string
 *               function_record:
 *                 type: string
 *               title:
 *                 type: string
 *               regulatory_body:
 *                 type: string
 *               legislation_sector_type:
 *                 type: string
 *               hierarchy_of_legislation:
 *                 type: string
 *               subordinate:
 *                 type: string
 *               heading:
 *                 type: string
 *               level_of_legislation:
 *                 type: string
 *               description_of_level:
 *                 type: string
 *               provision_of_the_requirement:
 *                 type: string
 *               interpretation_of_the_provision:
 *                 type: string
 *               comment_action_implication:
 *                 type: string
 *               consequences_of_non_compliance:
 *                 type: string
 *               organization:
 *                 type: string
 *             example:
 *               id: "1"
 *               risk_given_id: "R001"
 *               function_record: "Compliance"
 *               title: "Environmental Protection Act"
 *               regulatory_body: "Department of Environment"
 *               legislation_sector_type: "Environmental"
 *               hierarchy_of_legislation: "National"
 *               subordinate: "N/A"
 *               heading: "Environmental Protection"
 *               level_of_legislation: "Act"
 *               description_of_level: "Main legislation"
 *               provision_of_the_requirement: "Protect the environment"
 *               interpretation_of_the_provision: "Strict adherence"
 *               comment_action_implication: "High compliance requirements"
 *               consequences_of_non_compliance: "Fines and penalties"
 *               organization: "Org001"
 *     responses:
 *       200:
 *         description: Successfully created or updated the She Legal Register record
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *             example:
 *               message: "Legal register record successfully updated."
 */
sheLegalRegisterRoutes.post(
  "/update-she-legal-register",
  verifyToken,
  controllerWrapper(updateSheLegalRegister)
);

export default sheLegalRegisterRoutes;
