// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { createUpdateSkill, getSkill, deleteSkill } from "../controller/skillsController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { skillCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { validateRequest } from "../helper/general.js";

/**
 * @swagger
 * tags:
 *   name: skill
 *   description: API Operations For Managing All skill
 */
const SkillRoutes = Router();

/**
 * @swagger
 * /api/skill/create-update-skill:
 *   post:
 *     summary: Create Or Update A skill
 *     tags: [skill]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { "name": "name", "skill_type": "skill_type", "description": "description", "industry": "industry", "organization": "organization", "id": "id" }
 */
SkillRoutes.post(
  "/create-update-skill",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 91, "skills")(req, res, next);
  },
  validateRequest(skillCreateUpdateValidationSchema),
  controllerWrapper(createUpdateSkill)
);

/**
 * @swagger
 * /api/skill/get-skill:
 *   get:
 *     summary: Get All skill List
 *     tags: [skill]
 */
SkillRoutes.get("/get-skill", verifyToken, sideBarPermissionCheck("view", 91, "skills"), controllerWrapper(getSkill));

/**
 * @swagger
 * /api/skill/get-skill/{skillId}:
 *   get:
 *     summary: Get A Specific skill Record
 *     tags: [skill]
 */
SkillRoutes.get(
  "/get-skill/:id",
  verifyToken,
  sideBarPermissionCheck("view", 91, "skills"),
  controllerWrapper(getSkill)
);

/**
 * @swagger
 * /api/skill/delete-skill/{skillId}:
 *   delete:
 *     summary: Delete a Specific Record of skill
 *     tags: [skill]
 */
SkillRoutes.delete(
  "/delete-skill/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 91, "skills"),
  controllerWrapper(deleteSkill)
);

export default SkillRoutes;
