// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  approveRejectStrategicRiskIdentification,
  deleteStrategicRiskIdentification,
  strategicRiskIdentificationCreateUpdate,
  updateStrategicRiskIdentificationPriority,
  viewStrategicRiskIdentification,
  viewStrategicRiskIdentificationApprovalWorkflow,
} from "../controller/strategicRiskIdentificationController.js";
import { validateRequest } from "../helper/general.js";
import { strategicRiskIdentificationValidationSchema } from "../validationsRecord/validationSchema.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: strategicRiskIdentification
 *   description: API Operations For Managing strategic risk identification
 */
const strategicRiskIdentificationRoutes = Router();

/**
 * @swagger
 * /api/strategicRiskIdentification/create-update-strategic-risk-identification:
 *   post:
 *     summary: Create Or Update A strategic risk identification
 *     tags: [strategicRiskIdentification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: "id"
 *             risk_register_id: "risk_register_id"
 *             risk_owner: "risk_owner"
 *             risk_name: "risk_name"
 *             risk_description: "risk_description"
 *             impact_description: "impact_description"
 *             causes_description: "causes_description"
 *             inherent_likelihood: "inherent_likelihood"
 *             inherent_consequence: "inherent_consequence"
 *             inherent_risk_rating: "inherent_risk_rating"
 *             inherent_risk_ranking: "inherent_risk_ranking"
 *             control_id: "control_id"
 *             control_name: "control_name"
 *             control_description: "control_description"
 *             control_design_intent: "control_design_intent"
 *             control_effectiveness: "control_effectiveness"
 *             control_owner: "control_owner"
 *             risk_rating: "risk_rating"
 *             risk_ranking: "risk_ranking"
 *             opportunity_identification: "opportunity_identification"
 *             opportunity_description: "opportunity_description"
 *             priority_confirmation: "priority_confirmation"
 *             organization: "organization"
 *             department: "department"
 */
strategicRiskIdentificationRoutes.post(
  "/create-update-strategic-risk-identification",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      32,
      "strategic_risk_identification"
    )(req, res, next);
  },
  validateRequest(strategicRiskIdentificationValidationSchema),
  controllerWrapper(strategicRiskIdentificationCreateUpdate)
);

/**
 * @swagger
 * /api/strategicRiskIdentification/get-strategic-risk-identification:
 *   get:
 *     summary: Get list of all strategic risk identification
 *     tags: [strategicRiskIdentification]
 */
strategicRiskIdentificationRoutes.get(
  "/get-strategic-risk-identification",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk_identification"),
  controllerWrapper(viewStrategicRiskIdentification)
);

strategicRiskIdentificationRoutes.post(
  "/update-strategic-risk-identification-priority",
  verifyToken,
  sideBarPermissionCheck("edit", 32, "strategic_risk_identification"),
  controllerWrapper(updateStrategicRiskIdentificationPriority)
);

/**
 * @swagger
 * /api/strategicRiskIdentification/get-strategic-risk-identification/{id}:
 *   get:
 *     summary: Get A Specific strategic risk identification record
 *     tags: [strategicRiskIdentification]
 */
strategicRiskIdentificationRoutes.get(
  "/get-strategic-risk-identification/:id",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk_identification"),
  controllerWrapper(viewStrategicRiskIdentification)
);

strategicRiskIdentificationRoutes.get(
  "/get-strategic-risk-identification-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk_identification"),
  controllerWrapper(viewStrategicRiskIdentificationApprovalWorkflow)
);

/**
 * @swagger
 * /api/strategicRiskIdentification/delete-strategic-risk-identification/{id}:
 *   delete:
 *     summary: Delete a specific record of strategic risk identification
 *     tags: [strategicRiskIdentification]
 */
strategicRiskIdentificationRoutes.delete(
  "/delete-strategic-risk-identification/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 32, "strategic_risk_identification"),
  controllerWrapper(deleteStrategicRiskIdentification)
);

/**
 * @swagger
 * /api/strategicRiskIdentification/approve-reject-strategic-risk-identification:
 *   post:
 *     summary: Approve or reject strategic risk identification
 *     tags: [strategicRiskIdentification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             strategicRiskIdentificationId: "strategicRiskIdentificationId"
 *             status: "status"
 *             comment: "comment"
 */
strategicRiskIdentificationRoutes.post(
  "/approve-reject-strategic-risk-identification",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk_identification"),
  controllerWrapper(approveRejectStrategicRiskIdentification)
);

// /**
//  * @swagger
//  * /api/strategicRiskIdentification/get-strategic-risk-identification-prioritized/{id}:
//  *   get:
//  *     summary: Get A Specific strategic risk identification Prioritized record
//  *     tags: [strategicRiskIdentification]
//  */
// strategicRiskIdentificationRoutes.get(
//   "/get-strategic-risk-identification-prioritized/:id",
//   verifyToken,
//   sideBarPermissionCheck("view", 32, "strategic_risk_identification"),
//   viewStrategicRiskIdentificationPrioritized
// );

export default strategicRiskIdentificationRoutes;
