import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateTemplateCategory,
  deleteTemplateCategory,
  getParentTemplateCategoryList,
  viewTemplateCategory,
} from "../controller/templateCategoryController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const templateCategoryRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Template Category
 *   description: API operations for managing all Template Categories
 */

/**
 * @swagger
 * /api/templateCategory/create-update-template-category:
 *   post:
 *     summary: Create or update an Template Category
 *     tags: [Template Category]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: integer
 *           example:
 *             id: 1
 *             name: "Safety Incident"
 *             description: "Category for safety-related incidents"
 *             category : "category"
 *             sub_category: "sub category"
 *             organization: [1, 2]
 */
templateCategoryRoutes.post(
  "/create-update-template-category",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      249,
      "template_category"
    )(req, res, next);
  },
  controllerWrapper(createUpdateTemplateCategory)
);

/**
 * @swagger
 * /api/templateCategory/get-template-category:
 *   get:
 *     summary: Get list of all Incident Categories
 *     tags: [Template Category]
 */
templateCategoryRoutes.get(
  "/get-template-category",
  verifyToken,
  controllerWrapper(viewTemplateCategory)
);

/**
 * @swagger
 * /api/templateCategory/get-template-category/{id}:
 *   get:
 *     summary: Get a specific record of an Template Category
 *     tags: [Template Category]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Template Category to retrieve
 */
templateCategoryRoutes.get(
  "/get-template-category/:id",
  verifyToken,
  controllerWrapper(viewTemplateCategory)
);

/**
 * @swagger
 * /api/templateCategory/delete-template-category/{id}:
 *   delete:
 *     summary: Delete a specific record of an Template Category
 *     tags: [Template Category]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Template Category to delete
 */
templateCategoryRoutes.delete(
  "/delete-template-category/:id",
  verifyToken,
  controllerWrapper(deleteTemplateCategory)
);

/**
 * @swagger
 * /api/templateCategory/get-template-category-list:
 *   get:
 *     summary: Get the list for selecting parent category
 *     tags: [Template Category]
 *     parameters:
 *       - in: query
 *         organization : [1,2,34,4]
 *         schema:
 *           type: Array
 *         description: Array of organization to fetch the list
 */

templateCategoryRoutes.get(
  "/get-template-category-list",
  verifyToken,
  controllerWrapper(getParentTemplateCategoryList)
);

export default templateCategoryRoutes;
