import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";

import {
  createUpdateTrainingNeedAnalysis,
  getTrainingNeedAnalysis,
  deleteTrainingNeedAnalysis,
  approvedRejectTraining,
} from "../controller/trainingNeedAnalysisController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const trainingNeedAnalysisRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Training Need Analysis
 *   description: API operations for managing training need analysis
 */

/**
 * @swagger
 * /api/trainingNeedAnalysis/create-update-training-need-analysis:
 *   post:
 *     summary: Create or update a training need analysis
 *     tags: [Training Need Analysis]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/TrainingNeedAnalysis'
 *           example:
 *             organization : 1,
 *             training_need_analysis : [{name: "John",
 *             surname: "Doe",
 *             employee_id: "E12345",
 *             gender: "Male",
 *             disability: "None",
 *             race: "Caucasian",
 *             employee_type: "Full-time",
 *             role_name: "Manager",
 *             training_type: "Leadership",
 *             training_program: "Advanced Leadership Training",
 *             training_purpose: "Enhance leadership skills",
 *             approval_status: "Pending"}]
 *     responses:
 *       200:
 *         description: The created or updated training need analysis
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/TrainingNeedAnalysis'
 */
trainingNeedAnalysisRoutes.post(
  "/create-update-training-need-analysis",
  verifyToken,
  (req , res, next)=>{
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 197, "training_need_analysis")(req, res, next);
  },
  controllerWrapper(createUpdateTrainingNeedAnalysis)
);

/**
 * @swagger
 * /api/trainingNeedAnalysis/get-training-need-analysis:
 *   get:
 *     summary: Get all training need analyses
 *     tags: [Training Need Analysis]
 *     responses:
 *       200:
 *         description: A list of training need analyses
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/TrainingNeedAnalysis'
 */
trainingNeedAnalysisRoutes.get(
  "/get-training-need-analysis",
  verifyToken,
  sideBarPermissionCheck("view", 197, "training_need_analysis"),
  controllerWrapper(getTrainingNeedAnalysis)
);

/**
 * @swagger
 * /api/trainingNeedAnalysis/get-training-need-analysis/{id}:
 *   get:
 *     summary: Get a training need analysis by ID
 *     tags: [Training Need Analysis]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the training need analysis
 *     responses:
 *       200:
 *         description: The requested training need analysis
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/TrainingNeedAnalysis'
 */
trainingNeedAnalysisRoutes.get(
  "/get-training-need-analysis/:id",
  verifyToken,
  sideBarPermissionCheck("view", 197, "training_need_analysis"),
  controllerWrapper(getTrainingNeedAnalysis)
);

/**
 * @swagger
 * /api/trainingNeedAnalysis/delete-training-need-analysis/{id}:
 *   delete:
 *     summary: Delete a training need analysis by ID
 *     tags: [Training Need Analysis]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the training need analysis
 *     responses:
 *       200:
 *         description: The training need analysis was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Training need analysis deleted successfully"
 */
trainingNeedAnalysisRoutes.delete(
  "/delete-training-need-analysis/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 197, "training_need_analysis"),
  controllerWrapper(deleteTrainingNeedAnalysis)
);

/**
 * @swagger
 * /api/trainingNeedAnalysis/approve-reject-training:
 *   delete:
 *     summary: Approve or reject  a training need analysis by ID
 *     tags: [Training Need Analysis]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/TrainingNeedAnalysis'
 *           example:
 *             id : 1,
 *             status : approved/rejected
 *             comments: "comments"
 *     responses:
 *       200:
 *         description: The training need analysis was approved/rejected
 *         content:
 *           application/json:
 *             example:
 *               message: "Training need analysis approved or rejected successfully"
 */

trainingNeedAnalysisRoutes.post(
  "/approve-reject-training",
  verifyToken,
  controllerWrapper(approvedRejectTraining)
);

export default trainingNeedAnalysisRoutes;
