import { Router } from "express";
import {
  AuthOnlyUsersData,
  GetActiveLoginEmployeeData,
  GetEmployeeData,
  createEmployee,
  deleteEmployee,
  forgotPassword,
  forgotPasswordOTPVerify,
  getOrganizationChart,
  importEmployeeData,
  login,
  passwordChange,
  updateCredentials,
  updateEmployee,
  userPasswordChange,
} from "../controller/usersController.js";
import { validateRequest } from "../helper/general.js";

import {
  createEmployeeSchema,
  forgotPasswordValidationSchema,
  otpVerifyValidationSchema,
  passwordChangeValidationSchema,
  updateEmployeeSchema,
  userLoginSchema,
} from "../validationsRecord/validationSchema.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
import rateLimiter from "../middleware/rateLimiting.js";

/**
 * @swagger
 * tags:
 *   name: Users
 *   description: API operations for managing users
 */
const usersRoutes = Router();

/**
 * @swagger
 * /api/users/login:
 *   post:
 *     summary: User Login
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { field: "Email Address/Employee ID", password: "Password" }
 */
usersRoutes.post(
  "/login",
  validateRequest(userLoginSchema),
  // rateLimiter,
  controllerWrapper(login)
);

/**
 * @swagger
 * /api/users/forgot-password:
 *   post:
 *     summary: User Forgot Password
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { field: "Email Address/Employee ID" }
 */
usersRoutes.post(
  "/forgot-password",
  validateRequest(forgotPasswordValidationSchema),
  controllerWrapper(forgotPassword)
);

/**
 * @swagger
 * /api/users/forgot-password-otp-verify:
 *   post:
 *     summary: User Forgot Password OTP Verification
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { field: "Email Address/Employee ID", otp: "123456" }
 */
usersRoutes.post(
  "/forgot-password-otp-verify",
  validateRequest(otpVerifyValidationSchema),
  controllerWrapper(forgotPasswordOTPVerify)
);

/**
 * @swagger
 * /api/users/password-change:
 *   post:
 *     summary: User Password Changes
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { field: "Email Address/Employee ID", password: "123456", oldPassword: "123456789" }
 */
usersRoutes.post(
  "/password-change",
  validateRequest(passwordChangeValidationSchema),
  controllerWrapper(passwordChange)
);

/**
 * @swagger
 * /api/users/update-credential:
 *   post:
 *     summary: User Password Changes
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { field: "Email Address/Employee ID", password: "123456", oldPassword: "123456789" }
 */
usersRoutes.post(
  "/update-credential",
  validateRequest(passwordChangeValidationSchema),
  controllerWrapper(updateCredentials)
);

/**
 * @swagger
 * /api/users/get-employee-data:
 *   get:
 *     summary: Get All employee Record
 *     tags: [Users]
 */
usersRoutes.get(
  "/get-employee-data",
  verifyToken,
  sideBarPermissionCheck("view", 16, "users", "my_organization"),
  controllerWrapper(GetEmployeeData)
);

/**
 * @swagger
 * /api/users/auth-only:
 *   get:
 *     summary: Get All employee Record
 *     tags: [Users]
 */
usersRoutes.get(
  "/auth-only",
  verifyToken,
  sideBarPermissionCheck("view", 16, "users", "my_organization"),
  controllerWrapper(AuthOnlyUsersData)
);

/**
 * @swagger
 * /api/users/get-employee-data/{id}:
 *   get:
 *     summary: Get A Specific employee Record
 *     tags: [Users]
 */
usersRoutes.get(
  "/get-employee-data/:id",
  verifyToken,
  sideBarPermissionCheck("view", 16, "users", "my_organization"),
  controllerWrapper(GetEmployeeData)
);

/**
 * @swagger
 * /api/users/update-employee:
 *   post:
 *     summary: Update Employee
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             businessStructure: "1"
 *             employeeId: "employeeId"
 *             first_name: "first_name"
 *             surname: "surname"
 *             email: "email@gmail.com"
 *             password: "password"
 *             phone: "phone"
 *             id_number: "id_number"
 *             joining_date: "joining_date"
 *             end_date: "end_date"
 *             gender: "gender"
 *             disability: "disability"
 *             race: "race"
 *             employee_type: "employee_type"
 *             highest_qualification: "highest_qualification"
 *             name_of_qualification: "name_of_qualification"
 *             country: "country"
 *             state: "state"
 *             city: "city"
 *             current_add: "current_add"
 *             role: 1
 *             role_desc: "role_desc"
 *             hierarchy: "hierarchy"
 *             level: "level"
 *             accountability: "accountability"
 *             profile: "profile"
 *             id: "id"
 */
usersRoutes.post(
  "/update-employee",
  verifyToken,
  // sideBarPermissionCheck("edit", 16, "users", "my_organization"),
  // validateRequest(updateEmployeeSchema),
  controllerWrapper(updateEmployee)
);

/**
 * @swagger
 * /api/users/create-employee:
 *   post:
 *     summary: Create Employee
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             businessStructure: "1"
 *             employeeId: "employeeId"
 *             first_name: "first_name"
 *             surname: "surname"
 *             email: "email@gmail.com"
 *             password: "password"
 *             phone: "phone"
 *             license: "license"
 *             id_number: "id_number"
 *             joining_date: "joining_date"
 *             end_date: "end_date"
 *             gender: "gender"
 *             disability: "disability"
 *             race: "race"
 *             employee_type: "employee_type"
 *             highest_qualification: "highest_qualification"
 *             name_of_qualification: "name_of_qualification"
 *             country: "country"
 *             state: "state"
 *             city: "city"
 *             current_add: "current_add"
 *             role: 1
 *             role_desc: "role_desc"
 *             hierarchy: "hierarchy"
 *             level: "level"
 *             accountability: "accountability"
 *             profile: "profile"
 */
usersRoutes.post(
  "/create-employee",
  verifyToken,
  sideBarPermissionCheck("create", 16, "users", "my_organization"),
  // validateRequest(createEmployeeSchema),
  controllerWrapper(createEmployee)
);

/**
 * @swagger
 * /api/users/user-password-change:
 *   post:
 *     summary: User Password Changes
 *     tags: [Users]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "email/username", old_password: "123456", new_password: "123456" }
 */
usersRoutes.post(
  "/user-password-change",
  verifyToken,
  controllerWrapper(userPasswordChange)
);

/**
 * @swagger
 * /api/users/get-organization-employee-data:
 *   get:
 *     summary: Get All employee Record
 *     tags: [Users]
 */
usersRoutes.get(
  "/get-organization-employee-data",
  verifyToken,
  sideBarPermissionCheck("view", 16, "users", "my_organization"),
  controllerWrapper(getOrganizationChart)
);

/**
 * @swagger
 * /api/users/get-login-active-employee-data:
 *   get:
 *     summary: Get All employee Record
 *     tags: [Users]
 */
usersRoutes.get(
  "/get-login-active-employee-data",
  verifyToken,
  sideBarPermissionCheck("view", 16, "users", "my_organization"),
  controllerWrapper(GetActiveLoginEmployeeData)
);

/**
 * @swagger
 * /api/users/import-employee-data:
 *   post:
 *     summary: Import Record Of Employee
 *     tags: [Users]
 */
usersRoutes.post(
  "/import-employee-data",
  verifyToken,
  sideBarPermissionCheck("create", 16, "users", "my_organization"),
  controllerWrapper(importEmployeeData)
);

/**
 * @swagger
 * /api/users/delete-employee/{employeeId}:
 *   delete:
 *     summary: Delete A Specific Record Of Employee
 *     tags: [Users]
 */
usersRoutes.delete(
  "/delete-employee/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 16, "users", "my_organization"),
  controllerWrapper(deleteEmployee)
);

export default usersRoutes;
