import axios from 'axios';
import moment from 'moment';

export function setupClientCrashHandler(superadminApiUrl) {

    // --------------------------------------------------------------------------
    // 1. Handling Uncaught Exceptions (Synchronous Errors)
    // --------------------------------------------------------------------------
    /**
     * 'uncaughtException' is emitted when an uncaught JavaScript exception
     * bubbles all the way back to the event loop. This generally means an error
     * occurred in your synchronous code that was not caught by a try/catch block.
     *
     * WARNING: Using 'uncaughtException' is discouraged for production applications.
     *          The Node.js documentation advises against it because after an
     *          'uncaughtException', the state of the application is unknown and
     *          continuing execution can lead to unpredictable behavior and data corruption.
     *
     * However, for logging, alerting, and graceful shutdown in controlled environments,
     * it can be used carefully. In most cases, after handling 'uncaughtException',
     * it's best to terminate the process.
     */
    process.on('uncaughtException', async (err, origin) => {
        console.error('🚨🚨🚨 Uncaught Exception detected! Application is likely to crash soon.');
        console.error(`Exception origin: ${origin}`);
        console.error('Uncaught Exception:', err);

        // --- Task before actual "potential" crash ---
        await performPreCrashTasks(err, 'uncaughtException', superadminApiUrl);

        // --- Task after actual "potential" crash ---
        // Graceful shutdown or exit the process
        gracefulShutdown(1, 'uncaught exception.'); // Exit code 1 indicates an error
    });

    // --------------------------------------------------------------------------
    // 2. Handling Unhandled Promise Rejections (Asynchronous Errors)
    // --------------------------------------------------------------------------
    /**
     * 'unhandledRejection' is emitted when a Promise is rejected and no error handler
     * is attached to it within a turn of the event loop. This indicates an error
     * in your asynchronous code (Promises) that was not properly handled with `.catch()`
     * or a rejection handler in `async/await`.
     *
     * In modern Node.js versions, unhandled rejections can lead to process termination by default
     * (especially with `--unhandled-rejections=strict` or in future versions).
     *
     * Handling 'unhandledRejection' is crucial to prevent unexpected process exits
     * and to log/handle errors from asynchronous operations.
     */
    process.on('unhandledRejection', async (reason, promise) => {
        console.error('🚨🚨🚨 Unhandled Promise Rejection detected! Application might become unstable.');
        console.error('Unhandled Rejection at:', promise, 'reason:', reason);

        // --- Task before actual "potential" crash ---
        await performPreCrashTasks(reason, 'unhandledRejection', superadminApiUrl);

        // --- Task after actual "potential" crash ---
        // Graceful shutdown or exit the process
        gracefulShutdown(1, 'unhandled rejection.'); // Exit code 1 indicates an error
    });

    console.log('Client service: Crash handlers setup completed. 🚀'); // Confirmation log
}

/**
    * Function to perform tasks BEFORE a potential crash (e.g., logging, sending alerts).
    * This is called within 'uncaughtException' and 'unhandledRejection' handlers.
    * @param {Error|any} errorOrReason - The error object or rejection reason.
    * @param {string} eventType - The event type ('uncaughtException' or 'unhandledRejection').
    */
export async function performPreCrashTasks(errorOrReason, eventType, superadminApiUrl) {
    const crashLog = {
        timestamp: moment().format('YYYY-MM-DD HH:mm:ss'),
        eventType: eventType,
        errorDetails: errorOrReason instanceof Error ? {
            name: errorOrReason.name,
            message: errorOrReason.message,
            stack: errorOrReason.stack,
        } : errorOrReason, // Handle non-Error rejections if needed
    };

    console.warn('📧 Sending crash notification to Super Admin.....');
    const reportResult = await sendCrashReportToSuperadmin(crashLog, superadminApiUrl);

    if (reportResult.success) {
        console.log('Client service: Crash report process completed.');
        return;
    } else {
        console.error('Client service: Crash report process had issues:', reportResult.error);
        return;
    }
}


/**
    * Function to set up crash handlers for uncaughtException and unhandledRejection
    * in the client service.
    * @param {string} superadminApiUrl - The URL of the superadmin service's crash report API endpoint.
    */
async function sendCrashReportToSuperadmin(crashLog, superadminApiUrl) {
    try {
        const formattedPayload = {
            clientId: 1, // Dynamically retrieve the client ID
            alertType: ["System Failure"],
            alertSentDateTime: moment().format('YYYY-MM-DD HH:mm:ss'),
            deliveryStatus: "Pending",
            createdBy: 1, // Dynamically assign the user
            crashLog: crashLog
        };
        const fullUrl = `${superadminApiUrl}/api/notification-configuration-alert-log`;
        const response = await axios.post(fullUrl, formattedPayload);
        console.log('Client service: Crash report successfully sent to superadmin service.',
            { data: response.data }
        );
        return { success: true };
    } catch (apiError) {
        console.error('Client service: Error sending crash report to superadmin service:', apiError.message);
        return { success: false, message: apiError.message, error: apiError?.response?.data || [] };
    }
}

export function gracefulShutdown(exitCode, message) {
    console.log('Initiating graceful shutdown...');

    setTimeout(() => {
        console.log(`Client service: Exiting process with code ${exitCode} due to ${message}`);
        process.exit(exitCode);
    }, 2000); // Example timeout: 2 seconds to allow for cleanup. Adjust as needed.
}