import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  settingsUpdater,
  uniqueIdGenerator,
  whereCondition,
} from "../helper/general.js";
import AssessmentLocation from "../sequelize/AssessmentLocationSchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create new Assessment Location for employee  */
export const createUpdateAssessmentLocation = async (req, res) => {
  const { id, name, description, organization } = req.body;
  const encodedDescription = await encodeSingle_statement(description);

  /**If id comes in body then it will update the query */
  if (id) {
    if (organization.length === 0) {
      return sendResponse(
        res,
        200,
        "Please select at least one business structure"
      );
    }
    const result = await settingsUpdater(
      AssessmentLocation,
      organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 404, "No data found");
    }

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Assessment Location",
      `This user Update Assessment Location which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    /** Insert new record */
    const insertPromises = organization.map(async (element) => {
      // Check if the category type already exists for the organization
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM assessment_location WHERE name = ? AND deleted = "0" AND organization = ?`,
        [name, element]
      );

      // If record not found for that name with particular organization, insert the record
      if (checkNameWithOrganization.length === 0) {
         const unique_id = await uniqueIdGenerator(
            element,
            req.body?.department || null,
            "AssessmentLocation",
            "assessment_location",
            "unique_id",
            "unique_id"
          );
         
        // Insert Assessment Location details
        const insertAssessmentLocationQuery = `INSERT INTO assessment_location(name, description, created_by, organization,unique_id) VALUES (?, ?,?, ?, ?)`;
        await db.query(insertAssessmentLocationQuery, [
          name,
          encodedDescription,
          req.user.sessionid,
          element,
          unique_id
        ]);

        // Insert record for activity log
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "Assessment Location",
          `This user created a new category '${name}' for this organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Assessment Location */
export const viewAssessmentLocation = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "assessment_location",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchTableName = [
    "assessment_location.name",
    "assessment_location.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = assessment_location.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = assessment_location.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = assessment_location.department",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const AssessmentLocationQuery = `SELECT assessment_location.*,users.id AS created_by_id, users.name AS created_by, organization.name AS organization_name , users.surname as created_by_surname , users.profile as created_by_profile , department.name as department_name FROM assessment_location ${joinsRecord} where assessment_location.deleted = 0 ${searchCondition} ${condition}`;
  // return console.log(
  //   AssessmentLocationQuery,
  //   "AssessmentLocationQuery"
  // );
  let [AssessmentLocationRecord] = await db.query(AssessmentLocationQuery);

  for (const item of AssessmentLocationRecord) {
    item.description = await decodeSingle_statement(item.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM assessment_location WHERE deleted = 0 AND name = ?`,
        [item.name]
      );
      const arr = organizations.map((item) => item.organization);
      item.organizations = arr;
    }
  }

  /**Count all organization */
  const totalRecord = await countQueryCondition(AssessmentLocationQuery);

  return sendResponse(res, 200, AssessmentLocationRecord, totalRecord);
};

/**Function to delete a specific Assessment Location */
export const deleteAssessmentLocation = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord(
    "assessment_location",
    id
  );
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Assessment Location",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
