import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
  whereCondition,
  makeJoins,
  countQueryCondition,
  searchConditionRecord,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  encodeAndStringifyFields,
  insertActivityLog,
  uniqueIdGenerator,
} from "../helper/general.js";

import CompetencyAssessment from "../sequelize/CompetencyAssessmentSchema.js";

export const createUpdateCompetencyAssessment = async (req, res) => {
  const { id, department, individual_assessment } = req.body;
  // console.log(req.body);
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  if(!id){
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "CompetencyAssessment",
      "competency_assessment",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  for (const element of individual_assessment) {
    req.body.individual_assessment = [element];
    req.body = await encodeAndStringifyFields(req.body);
    const { query, values } = id
      ? updateQueryBuilder(CompetencyAssessment, req.body)
      : createQueryBuilder(CompetencyAssessment, req.body);
    await db.query(query, values);
  }
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Competency Assessment",
    `This user ${status} a new Competency Assessment Record`
  );
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getCompetencyAssessment = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "competency_assessment",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "competency_assessment.assessment_description",
    "competency_assessment.assessment_id",
    "competency_assessment.assessment_location",
    "competency_assessment.assessment_name",
    "competency_assessment.assessment_type",
    "competency_assessment.assessor_duration",
    "competency_assessment.assessor_method",
    "competency_assessment.competencies_assessed",
    "location.name",
    // "competency_assessment.training_purpose",
    // "competency_assessment.training_type",
    "assessor_method.name",
    "organization.name",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(accessor.name , ' '  , accessor.surname)",
    "roles.name",
    // "job.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = competency_assessment.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = competency_assessment.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = competency_assessment.department",
    },
    {
      type: "left",
      targetTable: "users AS accessor",
      onCondition: "accessor.id = competency_assessment.assessor_name",
    },
    {
      type: "left",
      targetTable: "gender",
      onCondition: "accessor.id = gender.id",
    },
    {
      type: "left",
      targetTable: "disability",
      onCondition: "accessor.id = disability.id",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = competency_assessment.job_role",
    },
    {
      type: "left",
      targetTable: "training_schedule",
      onCondition: "training_schedule.id = competency_assessment.schedule_training",
    },

    {
      type: "left",
      targetTable: "training_need_analysis",
      onCondition: "training_need_analysis.id = training_schedule.training_program",
    },

    {
      type: "left",
      targetTable: "location",
      onCondition:
        "location.id = competency_assessment.assessment_location",
    },
    {
      type: "left",
      targetTable: "assessor_method",
      onCondition: "assessor_method.id = competency_assessment.assessor_method",
    },
    {
      type: "left",
      targetTable: "competencies_assessed",
      onCondition: "competencies_assessed.id = competency_assessment.competencies_assessed",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const competencyAssessmentQuery = `SELECT competency_assessment.* ,competencies_assessed.name as competencies_assessed_name , gender.name AS gender_name, disability.name AS disability_name ,  organization.name as organization_name , department.name as department_name , users.name as created_by_name , users.surname AS created_by_surname , users.profile AS created_by_profile , accessor.name AS accessors_name , accessor.surname AS accessors_surname , accessor.profile AS accessors_profile , roles.name AS job_role_name ,
  training_need_analysis.training_name AS
  schedule_training_name ,
  location.name AS assessment_location_name , assessor_method.name as assessor_method_name   FROM competency_assessment ${joinCondition} WHERE competency_assessment.deleted = 0 ${searchCondition} ${condition} `;

  let [competencyAssessment] = await db.query(competencyAssessmentQuery);

  competencyAssessment = await decodeAndParseFields(competencyAssessment);

  for (const item of competencyAssessment) {
    if (item.individual_assessment) {
      for (const ele of item.individual_assessment) {
        if (ele.individual_type == "0") {
          const id = ele.name;
          const [dataFetch] = await db.query(
            `SELECT users.* ,  CONCAT(users.name , ' ' , users.surname) AS name , roles.name AS role_name , 
            gender.name AS gender_name , disability.name AS disability_name FROM users LEFT JOIN roles ON roles.id = users.role
            LEFT JOIN gender ON gender.id = users.gender
            LEFT JOIN disability ON disability.id = users.disability WHERE users.id = ${id}`
          );
          ele.employee_name = dataFetch[0]?.name;
          ele.employee_profile = dataFetch[0]?.profile;
          ele.employee_id = dataFetch[0]?.unique_id;
          ele.gender = dataFetch[0]?.gender;
          ele.gender_name = dataFetch[0]?.gender_name;
          ele.disability = dataFetch[0]?.disability;
          ele.disability_name = dataFetch[0]?.disability_name;
          ele.race = dataFetch[0]?.race;
          ele.employee_type = dataFetch[0]?.employee_type;
        }
        if (ele.training_program) {
          const [fetchTraining] = await db.query(
            `SELECT training_need_analysis.training_type , purpose_of_training.name AS training_purpose , training_need_analysis.training_name AS training_program_name FROM training_need_analysis LEFT JOIN purpose_of_training ON purpose_of_training.id = training_need_analysis.training_purpose  WHERE training_need_analysis.id = ${Number(
              ele.training_program
            )}`
          );
          ele.training_program_name = fetchTraining[0]?.training_program_name;
          ele.training_type = fetchTraining[0]?.training_type;
          ele.training_purpose = fetchTraining[0]?.training_purpose;
        }
      }
    }
  }

  const totalRecord = await countQueryCondition(competencyAssessmentQuery);

  return sendResponse(res, 200, competencyAssessment, totalRecord);
};

export const deleteCompetencyAssessment = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("competency_assessment", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Competency Assessment",
    `This user deleted a new Competency Assessment Record for organization ${id}`
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
