import db from "../db-config.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";
import {
  makeJoins,
  whereCondition,
  searchConditionRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  updateQueryBuilder,
  createQueryBuilder,
  insertActivityLog,
  decodeAndParseFields,
  countQueryCondition,
  deleteRecord,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorAppointment from "../sequelize/ContractorAppointmentSchema.js";

export const createUpdateContractorAppointment = async (req, res) => {
  const { id, department } = req.body;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  req.body = await encodeAndStringifyFields(req.body);

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ContractorAppointment",
      "contractor_appointments",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(ContractorAppointment, req.body)
    : createQueryBuilder(ContractorAppointment, req.body);

  const [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Contractor Appointment",
    id ? id : result.insertId
  );
  return sendResponse(res, 200, `Record  ${status} Successfully`);
};

export const getContractorAppointment = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "contractor_appointments",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    // "contractor_appointments.appoints_contractor_employee_name",
    // "contractor_appointments.appoints_contractor_employee_surname",
    // "contractor_appointments.appoints_contractor_employee_id",
    // "contractor_appointments.appoints_contractor_employee_designation",
    "organization.name",
    // "department.name",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
    // "roles.name",
    "appointment_type.name",
    "contractor_registration.contractor_name",
    "project_registration.project_title",
    "appoints_role.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = contractor_appointments.created_by",
    },
    {
      type: "left",
      targetTable: "users as appoints_user",
      onCondition:
        "appoints_user.id = contractor_appointments.appoints_user_id",
    },
    {
      type: "left",
      targetTable: "department as appoints_department",
      onCondition: "appoints_department.id = appoints_user.department",
    },
    {
      type: "left",
      targetTable: "roles as appoints_role",
      onCondition: "appoints_role.id = appoints_user.role",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = contractor_appointments.physical_location",
    },
    {
      type: "left",
      targetTable: "appointment_type",
      onCondition:
        "appointment_type.id = contractor_appointments.name_of_appointment",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_appointments.organization",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition:
        "contractor_registration.id = contractor_appointments.contractor",
    },

    {
      type: "left",
      targetTable: "project_registration",
      onCondition: "project_registration.id = contractor_appointments.project",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: " repository.id = contractor_appointments.ddrm_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const contractorAppointmentQuery = `SELECT contractor_appointments.* ,  contractor_registration.contractor_name AS contractor_name ,repository.url as signature, contractor_registration.cipc_registration_number AS contractor_cipc_registration_number, contractor_registration.contractor_employees AS contractor_employees  , appointment_type.name AS name_of_appointment_name , appointment_type.assignment AS appointment_assignment , appointment_type.designated AS designated_function , appointment_type.applicable AS applicable_compliance_obligation , CONCAT(appoints_user.name , ' ' , appoints_user.surname) AS appoints_user_name , appoints_user.profile AS appoints_user_profile , appoints_user.unique_id AS appoints_user_employee_number , appoints_department.name AS appoints_department_name , appoints_role.name AS appoints_role_name , location.name AS physical_location_name ,  CONCAT(createdUser.name , ' ' , createdUser.surname) AS created_by_name , createdUser.profile AS created_by_profile , organization.name AS organization_name , project_registration.project_title AS project_name FROM contractor_appointments  ${joinCondition} WHERE contractor_appointments.deleted = 0 ${searchCondition} ${condition}`;

  let [contractorAppointment] = await db.query(contractorAppointmentQuery);

  contractorAppointment = await decodeAndParseFields(contractorAppointment);

  const totalRecord = await countQueryCondition(contractorAppointmentQuery);
  return sendResponse(res, 200, contractorAppointment, totalRecord);
};

export const deleteContractorAppointment = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(ContractorAppointment, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Contractor Appointment",
    id
  );
  return sendResponse(res, 200, "Record Deleted Successfully");
};

export const uploadSignature = async (req, res) => {
  const { ddrm_id, id } = req.body;
  // const signature = req.files?.signature;
  if (!ddrm_id) {
    return sendResponse(res, 400, "ddrm_id is required");
  }

  // const ddrm_id = await uploadToDDRM(sidebar_id, signature, req);

  const { query, values } = updateQueryBuilder(ContractorAppointment, {
    id,
    ddrm_id,
  });

  const [result] = await db.query(query, values);
  if (result.affectedRows > 0) {
    return sendResponse(res, 200, "Signature uploaded successfully");
  } else {
    return sendResponse(res, 400, "Error uploading signature");
  }
};
