import db from "../db-config.js";
import {
  checkNameInOrganization,
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  processesSingleDDRMDocument,
  searchConditionRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorDocumentRepository from "../sequelize/ContractorDocumentRepositorySchema.js";

/**Function to create and update Contractor Document Repository  */
export const createUpdatedContractorDocumentRepository = async (req, res) => {
  const {
    id,
    document_name,
    document_description,
    document_path,
    organization,
    department,
    sidebar_id = 280,
  } = req.body;

  req.body.document_description = await encodeSingle_statement(
    document_description
  );
  // req.body.allowed_access_to = JSON.stringify(req.body.allowed_access_to);

  const document = req.files && req.files.document;

  if (document) {
    // req.body.document_path = await uploadFile(
    //   "contractor_document_repository",
    //   document
    // );
    const ddrm_id = await processesSingleDDRMDocument(
      "contractor_document_repository",
      sidebar_id,
      document,
      req
    );
    req.body.ddrm_id = ddrm_id;
  }

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**Check that this organization have that particular name or not*/
  const checkNameWithOrganization = await checkNameInOrganization(
    "contractor_document_repository",
    document_name,
    organizationId,
    "document_name"
  );

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    if (
      checkNameWithOrganization.length > 0 &&
      checkNameWithOrganization[0]?.id != id
    ) {
      return sendResponse(
        res,
        400,
        `Record with  Document name ${document_name} already exists.`
      );
    }

    const { query, values } = updateQueryBuilder(
      ContractorDocumentRepository,
      req.body
    );
    await db.query(query, values);
    /** update the file path if document is updated */

    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "update",
      "Contractor Document Repository",
      `This user Update other Skills which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    if (checkNameWithOrganization.length > 0) {
      return sendResponse(res, 400, "This record is already exists");
    }
    req.body.created_by = req.user.sessionid;
    /** Insert Contractor Document Repository details */
    const { query, values } = createQueryBuilder(
      ContractorDocumentRepository,
      req.body
    );
    await db.query(query, values);
    /** Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Contractor Document Repository",
      `This user created a new other Skills named ${document_name} for organization ${organizationId}`
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Contractor Document Repository */
export const viewAllContractorDocumentRepository = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_document_repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "contractor_document_repository.document_name",
    "contractor_document_repository.document_description",
    "CONCAT(users.name , ' ' , users.surname)",
    "contractor_audit_scheduling.audit_name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_document_repository.created_by",
    },
    {
      type: "left",
      targetTable: "contractor_audit_scheduling",
      onCondition:
        "contractor_audit_scheduling.id = contractor_document_repository.associated_audit",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = contractor_document_repository.organization",
    },
    {
      type: "left",
      targetTable: "document_type",
      onCondition:
        "document_type.id = contractor_document_repository.document_type",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = contractor_document_repository.ddrm_id",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all Contractor Document Repository */
  const contractorDocumentRepositoryDataFetchQuery = `SELECT contractor_document_repository.id,contractor_document_repository.created_by as created_by_id,contractor_document_repository.created_at,contractor_document_repository.document_type,contractor_document_repository.associated_audit,contractor_document_repository.document_name,contractor_document_repository.document_description,contractor_document_repository.allowed_access_to,
    repository.url as document_path,
    contractor_audit_scheduling.name as associated_audit_name,
    document_type.name as document_type_name,CONCAT(users.name , ' ' , users.surname ) as created_by ,users.profile as created_by_profile,organization.name as organization_name,contractor_document_repository.organization
    FROM contractor_document_repository  
    ${joinsRecord} 
    where contractor_document_repository.deleted = 0  ${searchCondition} ${condition}`;
  let [contractorDocumentRepositoryDataFetch] = await db.query(
    contractorDocumentRepositoryDataFetchQuery
  );
  contractorDocumentRepositoryDataFetch = await decodeAndParseFields(
    contractorDocumentRepositoryDataFetch
  );

  for (const item of contractorDocumentRepositoryDataFetch) {
    const users = item.allowed_access_to;
    const arr = [];
    if (users && Array.isArray(users)) {
      for (const user of users) {
        const [fetchUser] = await db.query(
          `SELECT contractor_name AS name FROM contractor_registration WHERE id = ${user}`
        );
        const obj = {};
        obj.id = user;
        obj.name = fetchUser[0]?.name;
        arr.push(obj);
        item.allowed_access_to_name = arr;
      }
    }
  }

  /**Count all Contractor Document Repository */
  const totalRecord = await countQueryCondition(
    contractorDocumentRepositoryDataFetchQuery
  );

  return sendResponse(
    res,
    200,
    contractorDocumentRepositoryDataFetch,
    totalRecord
  );
};

/**Function to delete a specific Contractor Document Repository */
export const deleteContractorDocumentRepository = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord(
    "contractor_document_repository",
    id
  );
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "delete",
      "Contractor Document Repository",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
