import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  insertMeetingActionNotes,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import ContractorMeetingRecording from "../sequelize/ContractorMeetingRecordingSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorMeetingNotes from "../sequelize/ContractorNotesSchema.js";

/**Function to create and update Contractor Meeting Recording  */
export const createUpdateContractorMeetingRecording = async (req, res) => {
  const {
    id,
    scheduled_meeting,
    approval_status,
    meeting_notes,
    organization,
    department,
    send_for_approval,
  } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */

  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  let arr = [];
  let created_user = req.user.sessionid;
  if (send_for_approval) {
    if (created_user == 1) {
      req.body.for_approval = 1;
      req.body.approval_status = "Approved";
    } else {
      const [teams] = await db.query(`SELECT * FROM teams WHERE deleted = 0`);
      for (const team of teams) {
        let teamMembers = JSON.parse(team.team_members);
        // console.log(teamMembers);
        // console.log(team.team_leader);
        if (teamMembers.includes(created_user)) {
          arr.push(team.team_leader);
          break;
        }
      }
      if (arr.length == 0 || arr[0] == null) {
        req.body.for_approval = 1;
        req.body.approval_status = "Approved";
      } else {
        req.body.send_to_employee = arr;
        req.body.for_approval = 1;
      }
    }
  }

  /**If id comes in body then it will update the query */
  if (id) {
    if (approval_status) {
      await db.query(
        `UPDATE contractor_meeting_recording SET  status = '${approval_status}' WHERE id = ${id}`
      );
    } else {
      /**Update record for Contractor Meeting Recording */
      req.body.for_approval = req.body.send_for_approval ? 1 : 0;
      req.body.updated_by = req.user.sessionid;
      const { query, values } = updateQueryBuilder(
        ContractorMeetingRecording,
        req.body
      );

      await db.query(query, values);

      await insertMeetingActionNotes(
        scheduled_meeting,
        req.user.sessionid,
        meeting_notes,
        id,
        ContractorMeetingNotes
      );
    }
    if (arr.length > 0) {
      for (let i = 0; i < arr.length; i++) {
        const publicUrl = `contractor-meeting-recording/approval-workflow/view/${id}`;
        await insertNotification(
          "Training Need Analysis Approval",
          publicUrl,
          arr[i],
          "url",
          req.user.sessionid
        );
      }
    }

    /**Insert Activity  */
    insertActivityLog(
      req.user.sessionid,
      "update",
      "Contractor Meeting Recording ",
      id
    );
    await deleteRecord(ContractorMeetingNotes, id, "meeting_recording_id");
    await insertMeetingActionNotes(
      scheduled_meeting,
      req.user.sessionid,
      meeting_notes,
      id,
      ContractorMeetingNotes
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    req.body.created_by = req.user.sessionid;


    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ContractorRegistration",
      "contractor_registration",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  

    /**Insert record for Contractor Meeting Recording */
    const { query, values } = createQueryBuilder(
      ContractorMeetingRecording,
      req.body
    );
    const [createContractorMeetingRecording] = await db.query(query, values);

    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Contractor Meeting Recording ",
      createContractorMeetingRecording.insertId
    );

    if (arr.length > 0) {
      for (let i = 0; i < arr.length; i++) {
        const publicUrl = `contractor-meeting-recording/approval-workflow/view/${createContractorMeetingRecording.insertId}`;
        await insertNotification(
          "Training Need Analysis Approval",
          publicUrl,
          arr[i],
          "url",
          req.user.sessionid
        );
      }
    }

    await insertMeetingActionNotes(
      scheduled_meeting,
      req.user.sessionid,
      meeting_notes,
      createContractorMeetingRecording.insertId,
      ContractorMeetingNotes
    );

    return sendResponse(res, 200, "Record Created Successfully");
  }
};

/**Function to view all  contractor_meeting recording */
export const viewAllContractorMeetingRecording = async (req, res) => {
  const condition = await whereCondition({
    table: "contractor_meeting_recording",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "contractor_meeting_recording.status",
    "contractor_meeting.meeting_title",
    "contractor_meeting.venue",
    "contractor_meeting.meeting_hierarchy",
    "organization.name",
    "contractor_meeting.meeting_title",
    "contractor_meeting.meeting_title",
    "CONCAT(meetingScriber.name , ' ' , meetingScriber.surname)",
    "CONCAT(u1.name , ' ' , u1.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const filter = req.query.filter && JSON.parse(req.query.filter);

  let ownerCondition = "";
  if (filter?.for_approval == 1) {
    ownerCondition = req.user.isSuperAdmin
      ? ""
      : `AND JSON_CONTAINS(contractor_meeting_recording.send_to_employee, ${req.user.sessionid} )`;
  }

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_meeting_recording.created_by",
    },
    {
      type: "left",
      targetTable: "contractor_meeting",
      onCondition:
        "contractor_meeting.id = contractor_meeting_recording.scheduled_meeting",
    },
    {
      type: "left",
      targetTable: "users as meetingOwner",
      onCondition: "meetingOwner.id = contractor_meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "users as meetingScriber",
      onCondition:
        "meetingScriber.id = contractor_meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = contractor_meeting_recording.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_meeting_recording.department",
    },
    {
      type: "left",
      targetTable: "contractor_meeting_notes",
      onCondition:
        "contractor_meeting_notes.meeting_id = contractor_meeting_recording.department",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all meetings */
  const meetingsQuery = `SELECT contractor_meeting_recording.unique_id AS unique_id, contractor_meeting_recording.approval_status as approval_status , contractor_meeting_recording.participants , meetingScriber.name as meeting_scriber,meetingScriber.surname as meeting_scriber_surname,meetingScriber.profile as meeting_scriber_profile , contractor_meeting_recording.scheduled_meeting , contractor_meeting_recording.organization , contractor_meeting_recording.meeting_scriber as meeting_scriber_id ,  meetingOwner.name as meeting_owner , meetingOwner.surname as meeting_owner_surname , meetingOwner.profile as  meeting_owner_profile , contractor_meeting.meeting_title ,
    contractor_meeting.meeting_hierarchy,contractor_meeting_recording.actual_meeting_date_from ,contractor_meeting_recording.actual_meeting_date_to , contractor_meeting_recording.meeting_duration , contractor_meeting.venue , contractor_meeting.planned_meeting_date_from,contractor_meeting.planned_meeting_date_to,contractor_meeting.meeting_owner as meeting_owner_id , organization.name as organization_name, contractor_meeting_recording.id, contractor_meeting_recording.for_approval , contractor_meeting.meeting_id as meeting_id,contractor_meeting_recording.created_by as created_by_id,users.name as created_by ,users.surname as created_by_surname , users.profile as created_by_profile
     FROM contractor_meeting_recording 
     ${joinsRecord} where contractor_meeting_recording.deleted = 0 ${searchCondition} ${ownerCondition} ${condition}`;
  let [meetings] = await db.query(meetingsQuery);

  meetings = await decodeAndParseFields(meetings);

  /**Get contractor_meeting notes according to contractor_meeting id */
  for (const meetingsNotes of meetings) {
    const meetingNotesRecordQuery = `SELECT contractor_meeting_notes.raised_by as raised_by_id,contractor_meeting_notes.id,contractor_meeting_notes.agenda_item,contractor_meeting_notes.meeting_discussion_name,contractor_meeting_notes.meeting_discussion,contractor_meeting_notes.action_discussion_name,contractor_meeting_notes.meeting_action_items , contractor_meeting_notes.meeting_action_description , contractor_meeting_notes.assigned_to as assigned_to_id ,contractor_meeting_notes.due_date ,contractor_meeting_notes.meeting_action_progress ,contractor_meeting_notes.status , contractor_meeting_notes.notes_time ,raisedPerson.name as raised_by , raisedPerson.surname as raised_by_surname , raisedPerson.profile as raised_by_profile ,assignedToPerson.name as assigned_to , assignedToPerson.surname as assigned_to_surname , assignedToPerson.profile as assigned_to_profile
      FROM contractor_meeting_notes 
      left join users as raisedPerson on raisedPerson.id = contractor_meeting_notes.raised_by
      left join users as assignedToPerson on assignedToPerson.id = contractor_meeting_notes.assigned_to
      WHERE contractor_meeting_notes.meeting_recording_id = ${meetingsNotes.id} AND contractor_meeting_notes.deleted = 0`;
    let [meetingNotesRecord] = await db.query(meetingNotesRecordQuery);
    meetingNotesRecord = await decodeAndParseFields(meetingNotesRecord);
    meetingNotesRecord = await decodeAndParseFields(meetingNotesRecord);

    meetingsNotes.notes = meetingNotesRecord;

    /** Structuring for contractor_meeting participants */

    const participantsArray = [];

    if (meetingsNotes.participants) {
      for (const participant of meetingsNotes.participants) {
        const [user] = await db.query(`SELECT users.name, 
            users.id, 
            users.surname, 
            users.phone,
            users.unique_id,
            roles.name as role_name,
            organization.name as organization_name,
            users.profile FROM users
            LEFT JOIN roles ON roles.id = users.role
            LEFT JOIN organization ON organization.id = users.organization
            where users.id = ${participant.id} AND users.deleted = "0"`);
        participantsArray.push({ ...participant, ...user[0] });
      }
    }
    else{
      meetingsNotes.participants = [];
    }

    meetingsNotes.participants = participantsArray;
  }
  const result = meetings;

  const totalRecord = await countQueryCondition(meetingsQuery);

  return sendResponse(res, 200, result, totalRecord);
};

/**Function to delete a specific Meeting */
export const deleteContractorMeetingRecording = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord(
    "contractor_meeting_recording",
    id
  );
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "delete",
      "Contractor Meeting Recording",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**Function to view all  contractor_meeting recording */
export const viewAllContractorMeetingRecordingById = async (req, res) => {
  const condition = await whereCondition({
    table: "contractor_meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.meetingId,
    user: req.user,
    columnName: "meeting_id",
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "contractor_meeting.meeting_title",
    "contractor_meeting.venue",
    "contractor_meeting.meeting_hierarchy",
    "organization.name",
    "contractor_meeting.meeting_title",
    "contractor_meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "contractor_meeting_notes.meeting_discussion_name",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_meeting_recording.created_by",
    },
    {
      type: "left",
      targetTable: "contractor_meeting",
      onCondition:
        "contractor_meeting.id = contractor_meeting_recording.scheduled_meeting",
    },
    {
      type: "left",
      targetTable: "users as meetingOwner",
      onCondition: "meetingOwner.id = contractor_meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "users as meetingScriber",
      onCondition:
        "meetingScriber.id = contractor_meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = contractor_meeting_recording.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_meeting_recording.department",
    },
    {
      type: "left",
      targetTable: "contractor_meeting_notes",
      onCondition:
        "contractor_meeting_notes.meeting_id = contractor_meeting_recording.department",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all meetings */
  const meetingsQuery = `SELECT contractor_meeting.meeting_hierarchy,contractor_meeting_recording.participants , meetingScriber.name as meeting_scriber , contractor_meeting_recording.scheduled_meeting , contractor_meeting_recording.organization , contractor_meeting_recording.meeting_scriber as meeting_scriber_id ,  meetingOwner.name as meeting_owner , contractor_meeting.meeting_title ,contractor_meeting_recording.actual_meeting_date_from ,contractor_meeting_recording.actual_meeting_date_to , contractor_meeting_recording.meeting_duration , contractor_meeting.venue , contractor_meeting.planned_meeting_date_from,contractor_meeting.planned_meeting_date_to,contractor_meeting.meeting_owner as meeting_owner_id , organization.name as organization_name, contractor_meeting_recording.id,contractor_meeting.meeting_id as meeting_id,contractor_meeting_recording.created_by as created_by_id,users.name as created_by
     FROM contractor_meeting_recording 
     ${joinsRecord} where contractor_meeting_recording.deleted = 0 ${searchCondition} ${condition}`;
  const [meetings] = await db.query(meetingsQuery);

  /**Get contractor_meeting notes according to contractor_meeting id */

  for (const meetingsNotes of meetings) {
    const meetingNotesRecordQuery = `SELECT contractor_meeting_notes.raised_by as raised_by_id,contractor_meeting_notes.id,contractor_meeting_notes.agenda_item,contractor_meeting_notes.meeting_discussion_name,contractor_meeting_notes.meeting_discussion,contractor_meeting_notes.action_discussion_name,contractor_meeting_notes.meeting_action_items , contractor_meeting_notes.meeting_action_description , contractor_meeting_notes.assigned_to as assigned_to_id ,contractor_meeting_notes.due_date ,contractor_meeting_notes.meeting_action_progress ,contractor_meeting_notes.status , contractor_meeting_notes.notes_time ,raisedPerson.name as raised_by , raisedPerson.surname as raised_by_surname , raisedPerson.profile as raised_by_profile ,assignedToPerson.name as assigned_to , assignedToPerson.surname as assigned_to_surname , assignedToPerson.profile as assigned_to_profile
      FROM contractor_meeting_notes 
      left join users as raisedPerson on raisedPerson.id = contractor_meeting_notes.raised_by
      left join users as assignedToPerson on assignedToPerson.id = contractor_meeting_notes.assigned_to
      WHERE contractor_meeting_notes.contractor_meeting_recording_id = ${meetingsNotes.id}`;

    // const meetingNotesRecordQuery = `SELECT raised_by,id,agenda_item,meeting_discussion_name,meeting_discussion,action_discussion_name,meeting_action_items , meeting_action_description , assigned_to ,due_date ,meeting_action_progress ,status , notes_time
    //  FROM contractor_meeting_notes
    //  WHERE contractor_meeting_recording_id = ${meetingsNotes.id}`;
    const [meetingNotesRecord] = await db.query(meetingNotesRecordQuery);
    for (const notes of meetingNotesRecord) {
      notes.meeting_discussion = await decodeSingle_statement(
        notes.meeting_discussion
      );
      notes.meeting_action_description = await decodeSingle_statement(
        notes.meeting_action_description
      );
    }

    meetingsNotes.notes = meetingNotesRecord;

    /** Structuring for contractor_meeting participants */

    const participants = JSON.parse(meetingsNotes.participants);
    const participantsArray = [];

    for (const participant of participants) {
      const [user] = await db.query(`SELECT users.name, 
        users.id, 
        users.surname, 
        users.phone,
        roles.name as role_name,
        organization.name as organization_name,
        users.profile FROM users
        LEFT JOIN roles ON roles.id = users.role
        LEFT JOIN organization ON organization.id = users.organization
        where users.id = ${participant.id} AND users.deleted = "0"`);
      participantsArray.push({ ...participant, ...user[0] });
    }

    meetingsNotes.participants = participantsArray;
  }

  const result = meetings;
  const totalRecord = await countQueryCondition(meetingsQuery);
  return sendResponse(res, 200, result, totalRecord);
};

/**Function to view all  contractor_meeting recording */
export const getContractorMeetingRecordActionItem = async (req, res) => {
  const condition = await whereCondition({
    table: "contractor_meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "contractor_meeting.meeting_title",
    "contractor_meeting.venue",
    "contractor_meeting.meeting_hierarchy",
    "organization.name",
    "contractor_meeting.meeting_title",
    "contractor_meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "contractor_meeting_notes.meeting_discussion_name",
    "contractor_meeting_notes.meeting_discussion",
    "contractor_meeting_notes.meeting_action_items",
    "CONCAT(raised.name , ' ' , raised.surname)",
    "(meeting_owner.name , ' ' , meeting_owner.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_meeting_notes.assigned_to",
    },
    {
      type: "left",
      targetTable: "users as raised",
      onCondition: "raised.id = contractor_meeting_notes.raised_by",
    },
    {
      type: "left",
      targetTable: "contractor_meeting",
      onCondition:
        "contractor_meeting.id = contractor_meeting_notes.meeting_id",
    },
    {
      type: "left",
      targetTable: "contractor_meeting_recording",
      onCondition:
        "contractor_meeting_recording.scheduled_meeting = contractor_meeting_notes.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = contractor_meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "users as meeting_owner",
      onCondition: "meeting_owner.id = contractor_meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = contractor_meeting_recording.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all meetings */
  const meetingsNotesQuery = `SELECT contractor_meeting_recording.actual_meeting_date_from,
     contractor_meeting_recording.meeting_scriber,
     u1.name as meeting_scriber, 
     meeting_owner.name as meeting_chairperson,
     meeting_owner.surname as meeting_chairperson_surname,
     contractor_meeting.venue,
     contractor_meeting_notes.notes_time, raised.name as raised_by  , raised.surname as raised_by_surname , 
     raised.profile as raised_by_profile, contractor_meeting_notes.raised_by as  raised_by_id ,  
     contractor_meeting.meeting_id,contractor_meeting.meeting_title,contractor_meeting_notes.id,
     contractor_meeting_notes.agenda_item,contractor_meeting_notes.meeting_discussion_name,
     contractor_meeting_notes.completed_date,
     contractor_meeting_recording.organization,
     organization.name as organization_name,
     contractor_meeting_notes.meeting_discussion,contractor_meeting_notes.action_discussion_name,contractor_meeting_notes.meeting_action_items ,contractor_meeting_notes. meeting_action_description ,contractor_meeting_notes. assigned_to as assigned_to_id , users.name as assigned_to , users.surname as assigned_to_surname ,users.profile as assigned_to_profile ,contractor_meeting_notes.due_date ,contractor_meeting_notes.meeting_action_progress ,contractor_meeting_notes.status FROM contractor_meeting_notes
     ${joinsRecord} where contractor_meeting_notes.deleted = 0 ${searchCondition} ${condition}`;
  const [meetingsNotes] = await db.query(meetingsNotesQuery);
  for (const notes of meetingsNotes) {
    notes.meeting_discussion = await decodeSingle_statement(
      notes.meeting_discussion
    );
    notes.meeting_action_description = await decodeSingle_statement(
      notes.meeting_action_description
    );
  }
  const totalRecord = await countQueryCondition(meetingsNotesQuery);
  return sendResponse(res, 200, meetingsNotes, totalRecord);
};

/**Function to update notes status */
export const meetingNotesStatusChange = async (req, res) => {
  const { id, meeting_action_progress, status } = req.body;

  // await db.query(
  //   `UPDATE contractor_meeting_notes SET  status = '${status}' , meeting_action_progress = '${meeting_action_progress}' WHERE id = ${id}`
  // );
  await db.query(
    `UPDATE contractor_meeting_notes SET  status = '${status}', ${
      status === "Completed" ? "completed_date = NOW()," : ""
    } meeting_action_progress = '${meeting_action_progress}' WHERE id = ${id}`
  );

  return sendResponse(res, 200, "Record updated successfully");
};

export const updateApprovalStatus = async (req, res) => {
  /**Select all contractor_meeting approval record data  */
  const [getMeetingApprovalRecord] = await db.query(
    `SELECT id,due_date FROM contractor_meeting_notes WHERE deleted = "0" AND status != "Completed" AND due_date != "0000-00-00"`
  );

  // Function to get status based on due date
  async function getStatus(dueDate) {
    const timeDifference = dueDate.getTime() - Date.now();
    return timeDifference > 24 * 60 * 60 * 1000 ? "overdue" : "not overdue";
  }

  // Update status for records where due date is more than 24 hours from current date
  await Promise.all(
    getMeetingApprovalRecord.map(async (record) => {
      const status = await getStatus(new Date(record.due_date));
      if (status === "overdue") {
        await db.query(
          `UPDATE contractor_meeting_notes SET status = "overdue" WHERE id = ?`,
          [record.id]
        );
      }
    })
  );

  return sendResponse(res, 200, "Record updated successfully");
};
