import db from "../db-config.js";
import {
  createQueryBuilder,
  insertActivityLog,
  makeJoins,
  whereCondition,
  searchConditionRecord,
  countQueryCondition,
  updateQueryBuilder,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  deleteRecord,
  uniqueIdGenerator,
} from "../helper/general.js";
import ContractorRiskAssessment from "../sequelize/ContractorRiskAssessmentSchema.js";
import { sendResponse } from "../helper/wrapper.js";

export const createUpdateContractorRiskAssessment = async (req, res) => {
  const { id, department, risk_assessment } = req.body;
  let status = id ? "Updated" : "Created";

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  for (let risk of risk_assessment) {
    risk.id = req.body.id;
    risk.contractor = req.body.contractor;
    risk.department = req.body.department;
    risk.organization = req.body.organization;
    risk.project = req.body.project;
    risk[id ? "updated_by" : "created_by"] = req.user?.sessionid;
    risk = await encodeAndStringifyFields(risk);

    if(!id){
      const unique_id = await uniqueIdGenerator(
        req.body.organization ,
        department,
        "ContractorRiskAssessment",
        "contractor_risk_assessment",
        "unique_id",
        "unique_id"
      );
      
      risk.unique_id = unique_id;
    }

    // console.log(risk);
    let { query, values } = id
      ? updateQueryBuilder(ContractorRiskAssessment, risk)
      : createQueryBuilder(ContractorRiskAssessment, risk);

    let [result] = await db.query(query, values);

    await insertActivityLog(
      req.user.sessionid,
      status,
      "Contractor Risk Assessment",
      id ? id : result.insertId
    );
  }

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getContractorRiskAssessment = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_risk_assessment",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "contractor_risk_assessment.risk_name",
    "contractor_risk_assessment.risk_description",
    "contractor_risk_assessment.impact_description",
    "contractor_risk_assessment.causes_description",
    "contractor_risk_assessment.inherent_likelihood",
    "contractor_risk_assessment.inherent_consequence",
    "contractor_risk_assessment.inherent_risk_rating",
    "contractor_risk_assessment.inherent_risk_ranking",
    "contractor_risk_assessment.control_name",
    "contractor_risk_assessment.control_design_intent",
    "contractor_risk_assessment.control_effectiveness",
    "contractor_risk_assessment.control_description",
    "contractor_risk_assessment.residual_risk_rating",
    "contractor_risk_assessment.residual_risk_ranking",
    "contractor_risk_assessment.opportunity_description",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
    "organization.name",
    "department.name",
    "contractor_risk_register.risk_register_name",
    "project_registration.project_title",
    "cr.contractor_name",
    // "CONCAT(riskOwner.name , ' ' , riskOwner.surname)",
    // "CONCAT(controlOwner.name , ' ' , controlOwner.surname)",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = contractor_risk_assessment.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_risk_assessment.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_risk_assessment.department",
    },
    {
      type: "left",
      targetTable: "contractor_registration AS cr",
      onCondition: "cr.id = contractor_risk_assessment.contractor",
    },
    {
      type: "left",
      targetTable: "contractor_risk_register",
      onCondition:
        "contractor_risk_register.id = contractor_risk_assessment.risk_register",
    },
    {
      type: "left",
      targetTable: "project_registration",
      onCondition:
        "project_registration.id = contractor_risk_assessment.project",
    },
  ];
  const joinCondition = await makeJoins(joins);

  const contractorRiskAssessmentQuery = `SELECT contractor_risk_assessment.* , organization.name AS organization_name , department.name AS department_name , CONCAT(createdUser.name , ' ' , createdUser.surname) AS created_by_name,contractor_risk_register.risk_register_name AS risk_register_name,  createdUser.profile AS created_by_profile , cr.contractor_name AS contractor_name , cr.cipc_registration_number AS contractor_cipc_registration_number , project_registration.project_title AS project_name  FROM contractor_risk_assessment ${joinCondition} WHERE contractor_risk_assessment.deleted = 0 ${searchCondition} ${condition} `;
  //CONCAT(riskOwner.name , ' ' , riskOwner.surname) AS risk_owner_name, riskOwner.profile AS risk_owner_profile , CONCAT(controlOwner.name , ' ' , controlOwner.surname) AS control_owner_name , controlOwner.profile AS control_owner_profile

  let [contractorRiskAssessment] = await db.query(
    contractorRiskAssessmentQuery
  );
  contractorRiskAssessment = await decodeAndParseFields(
    contractorRiskAssessment
  );
  const totalRecord = await countQueryCondition(contractorRiskAssessmentQuery);

  return sendResponse(res, 200, contractorRiskAssessment, totalRecord);
};

export const deleteContractorRiskAssessment = async (req, res) => {
  const { id } = req.params;

  await deleteRecord(ContractorRiskAssessment, id);

  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Contractor Risk Assessment",
    id
  );

  return sendResponse(res, 200, "Record deleted successfully");
};
