import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  settingsUpdater,
  uniqueIdGenerator,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import DocumentCategories from "../sequelize/DocumentCategoriesSchema.js";

export const createUpdateCategories = async (req, res) => {
  const { id, name, description, organization, department } = req.body;
  const encodedDescription = await encodeSingle_statement(description);

  if (id) {
    if (organization.length === 0) {
      return sendResponse(
        res,
        200,
        "Please select at least one business structure"
      );
    }

    const result = await settingsUpdater(
      DocumentCategories,
      organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 404, "No data found");
    }

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Document Categories",
      `This user Update Document Categories with id  ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "DC",
      "document_categories",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    const insertPromises = organization.map(async (element) => {
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM document_categories WHERE name = ? AND deleted = "0" AND organization = ?`,
        [name, element]
      );

      if (checkNameWithOrganization.length === 0) {
        const insertQuery = `INSERT INTO document_categories(name, description, created_by, organization, unique_id) VALUES (?, ?, ?, ?, ?)`;
        await db.query(insertQuery, [
          name,
          encodedDescription,
          req.user.sessionid,
          element,
          req.body.unique_id,
        ]);

        // Insert record for activity log
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "Document Categories",
          `This user created a new category '${name}' for this organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);
    return sendResponse(res, 200, "Record created successfully");
  }
};

export const getCategories = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "document_categories",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchTableName = [
    "document_categories.name",
    "document_categories.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];

  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = document_categories.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = document_categories.organization",
    },
    // {
    //   type: "left",
    //   targetTable: "department",
    //   onCondition: "department.id = document_categories.department",
    // },
  ];
  const joinsRecord = await makeJoins(joins);

  const CategoriesQuery = `SELECT document_categories.*,users.id AS created_by_id, users.name AS created_by, organization.name AS organization_name , users.surname as created_by_surname , users.profile as created_by_profile  
    FROM document_categories ${joinsRecord} where document_categories.deleted = 0 ${searchCondition} ${condition}`;

  let [CategoriesRecord] = await db.query(CategoriesQuery);

  for (const item of CategoriesRecord) {
    item.description = await decodeSingle_statement(item.description);

    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM document_categories WHERE deleted = 0 AND name = ?`,
        [item.name]
      );
      const arr = organizations.map((item) => item.organization);
      item.organizations = arr;
    }
  }

  const totalRecord = await countQueryCondition(CategoriesQuery);
  return sendResponse(res, 200, CategoriesRecord, totalRecord);
};

export const deleteCategories = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord(
    "document_categories",
    id
  );
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Document Categories",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
