import path from "path";
import moment from "moment";
import db from "../db-config.js";
import cron from "node-cron";
import {
  auditLogFunction,
  copyRecursiveAsync,
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeAndStringifyFields,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  getRecord,
  getUserListByIds,
  incrementVersion,
  insertActivityLog,
  insertNotification,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  uploadFile,
  uploadToAzure,
  uploadToGCP,
  uploadToS3,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import DocumentCreation from "../sequelize/DocumentCreationSchema.js";
import fs from "fs";
import Repository from "../sequelize/RepositorySchema.js";
import chalk from "chalk";
import { fileURLToPath } from "url";
import { newUploadToDDRM } from "../helper/ddrmUploader.js";
import axios from "axios";
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const superadminApiUrl = process.env.SUPER_ADMIN_BACKEND_URL; // Ensure this env var is set

// export const createUpdateDocument = async (req, res) => {
//   let {
//     id,
//     organization,
//     description,
//     document_name,
//     parentId,
//     department,
//     sidebar_id,
//     select_revised_document,
//   } = req.body;
//   req.body.description = await encodeSingle_statement(description);
//   let organizationId = organization;
//   if (department) {
//     const recordAccordingToOrganization =
//       await getOrganizationAccordingToDepartment(department);
//     organizationId = recordAccordingToOrganization[0].organization;
//     req.body.organization = organizationId;
//     organization = organizationId;
//   }
//   if (id) {
//     const doc = await getRecord("document_creation", "id", id);
//     if (doc.length === 0) {
//       return sendResponse(res, 400, "Document not found");
//     }
//     const modificationRightsEmployees = JSON.parse(
//       doc[0]?.modification_rights_employees || "[]"
//     );
//     if (
//       !modificationRightsEmployees.includes(req.user.sessionid) &&
//       !req.user.isSuperAdmin
//     )
//       return sendResponse(
//         res,
//         400,
//         `cannot edit '${currentFolder[0]?.name}': You do not have modification right.`
//       );
//   }

//   if (!id) {
//     req.body.unique_id = await uniqueIdGenerator(
//       organizationId,
//       department,
//       "DC",
//       "document_creation",
//       "unique_id",
//       "unique_id"
//     );
//   }

//   let version = "1.0.0";
//   if (select_revised_document && !id) {
//     const [document] = await db.query(
//       `SELECT version_number FROM document_creation WHERE id = ${select_revised_document}`
//     );

//     if (document.length === 0) {
//       return sendResponse(res, 400, "Document not found");
//     }
//     const [archiveRevisedDocument] = await db.query(
//       `UPDATE document_creation SET is_archived = 1 WHERE id = ${select_revised_document}`
//     );

//     // also update in repository with this document_creation_id
//     await db.query(
//       `UPDATE repository SET is_archived = 1 WHERE document_creation_id = ${select_revised_document}`
//     );
//     version = document[0]?.version_number
//       ? incrementVersion(document[0]?.version_number)
//       : "1.0.0";
//     // req.body.select_revised_document = select_revised_document;
//   }
//   req.body.version_number = version;

//   // return console.log("req.body: ", req.body);
//   let actualPath = "";
//   let folderPath = "";
//   if (req?.files?.upload_document) {
//     let file = req?.files?.upload_document;
//     if (parentId !== "null" && !sidebar_id) {
//       [folderPath] = await db.query(
//         `SELECT * FROM repository WHERE id = ${parentId} AND organization = ${organization} AND deleted = 0 AND type = 'folder'  LIMIT 1 `
//       );
//       console.log(
//         chalk.blue(
//           `SELECT * FROM repository WHERE  id = ${parentId} AND organization = ${organization} AND deleted =0  LIMIT 1 `
//         )
//       );
//       if ((!folderPath || folderPath.length === 0) && parentId) {
//         return sendResponse(
//           res,
//           400,
//           "No folder found for the provided parent ID and organization."
//         );
//       }
//     }

//     folderPath = folderPath == "" ? `public` : `public/${folderPath[0]?.url}`;
//     actualPath = path.join(process.cwd(), `${folderPath}`);

//     if (sidebar_id && isNaN(sidebar_id)) {
//       return sendResponse(res, 400, "Invalid sidebar id");
//     }

//     if (sidebar_id) {
//       const data = await newUploadToDDRM(
//         req?.files?.upload_document,
//         sidebar_id,
//         req
//       );
//       actualPath = data.uploadedPath;
//       parentId = data.parentId;
//       req.body.sidebar_id = sidebar_id;
//     }

//     const fileName = req?.files?.upload_document?.name;

//     if (!fs.existsSync(actualPath)) {
//       fs.mkdirSync(actualPath, { recursive: true }); // Only create folder, not the file
//       console.log(chalk.yellow(`Created folder: ${actualPath}`));
//     }
//     // Now move the file to the correct location (not to a folder named after document)
//     actualPath = path.join(actualPath, fileName);
//     await file.mv(actualPath); // move the file to the actual path with document name
//     console.log(
//       chalk.yellow(`File moved to: ${path.join(actualPath, fileName)}`)
//     );
//   }
//   console.log(chalk.cyan(actualPath));
//   const lastPublicIndex = actualPath.lastIndexOf("public");
//   let filePath = actualPath.substring(lastPublicIndex + "public".length);

//   if (actualPath) {
//     req.body.upload_document = filePath;
//   }

//   // return console.log(parentId);
//   let status = id ? "Updated" : "Created";
//   req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

//   if (!organization || organization?.length === 0) {
//     return sendResponse(
//       res,
//       200,
//       "Please select at least one business structure"
//     );
//   }

//   const isLocked = sidebar_id ? 1 : 0;
//   req.body = await encodeAndStringifyFields(req.body);
//   req.body.is_locked = isLocked;

//   const { query, values } = id
//     ? updateQueryBuilder(DocumentCreation, req.body)
//     : createQueryBuilder(DocumentCreation, req.body);

//   const [result] = await db.query(query, values);
//   await insertActivityLog(
//     req.user.sessionid,
//     status,
//     "Document Creation",
//     id ? id : result.insertId
//   );
//   let createdData;
//   if (req?.files?.upload_document) {
//     const { query: RepositoryQuery, values: RepositoryValues } =
//       createQueryBuilder(Repository, {
//         name: document_name,
//         url: filePath,
//         parent_id: parentId,
//         organization,
//         type: "file",
//         document_creation_id: id ? id : result.insertId,
//         created_by: id ? result?.created_by : req.user.sessionid,
//         updated_by: id ? req.user.sessionid : null,
//         is_locked: isLocked,
//         sidebar_id: sidebar_id ? sidebar_id : null,
//       });
//     [createdData] = await db.query(RepositoryQuery, RepositoryValues);
//   }

//   const data = {
//     organization: req?.body?.organization,
//     department: req?.body?.department,
//     repository_id: createdData?.insertId,
//     created_by: req.user.sessionid,
//     document_name: document_name,
//     action_type: id ? "edit" : "upload",
//     document_status: req?.body?.document_status,
//   };
//   // audit log for document
//   await auditLogFunction(data);

//   let resultData = {
//     ddrm_id: createdData?.insertId ? createdData?.insertId : id,
//     url: filePath,
//   };

//   return sidebar_id
//     ? sendResponse(res, 200, resultData, `Record ${status} Successfully`)
//     : sendResponse(res, 200, `Record ${status} Successfully`);

//   // return sendResponse(res, 200, sidebar_id && resultData, `Record  ${status} Successfully`);
// };

export const createUpdateDocument = async (req, res) => {
  let {
    id,
    organization,
    description,
    document_name,
    parentId,
    department,
    sidebar_id,
    select_revised_document,
    ddrm_id,
  } = req.body;
  req.body.description = await encodeSingle_statement(description);
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
    organization = organizationId;
  }
  if (id) {
    const doc = await getRecord("document_creation", "id", id);
    if (doc.length === 0) {
      return sendResponse(res, 400, "Document not found");
    }
    const modificationRightsEmployees = JSON.parse(
      doc[0]?.modification_rights_employees || "[]"
    );
    if (
      !modificationRightsEmployees.includes(req.user.sessionid) &&
      !req.user.isSuperAdmin
    )
      return sendResponse(
        res,
        400,
        `cannot edit '${doc[0]?.name}': You do not have modification right.`
      );
  }

  if (!id) {
    req.body.unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "DC",
      "document_creation",
      "unique_id",
      "unique_id"
    );
  }
  let version = "1.0.0";
  if (select_revised_document && !id) {
    const [document] = await db.query(
      `SELECT version_number FROM document_creation WHERE id = ${select_revised_document}`
    );

    if (document.length === 0) {
      return sendResponse(res, 400, "Document not found");
    }
    const [archiveRevisedDocument] = await db.query(
      `UPDATE document_creation SET is_archived = 1 WHERE id = ${select_revised_document}`
    );

    // also update in repository with this document_creation_id
    await db.query(
      `UPDATE repository SET is_archived = 1 WHERE document_creation_id = ${select_revised_document}`
    );
    version = document[0]?.version_number
      ? incrementVersion(document[0]?.version_number)
      : "1.0.0";
  }
  req.body.version_number = version;
  let actualPath = "";
  let folderPath = "";
  if (req?.files?.upload_document) {
    let file = req?.files?.upload_document;
    if (
      parentId &&
      parentId !== "null" &&
      parentId !== "undefined" &&
      !sidebar_id
    ) {
      [folderPath] = await db.query(
        `SELECT * FROM repository WHERE id = ${parentId} AND organization = ${organization} AND deleted = 0 AND type = 'folder'  LIMIT 1 `
      );
      console.log(
        chalk.blue(
          `SELECT * FROM repository WHERE  id = ${parentId} AND organization = ${organization} AND deleted =0  LIMIT 1 `
        )
      );
      if ((!folderPath || folderPath.length === 0) && parentId) {
        return sendResponse(
          res,
          400,
          "No folder found for the provided parent ID and organization."
        );
      }
    }

    folderPath = folderPath == "" ? `public` : `public/${folderPath[0]?.url}`;
    actualPath = path.join(process.cwd(), `${folderPath}`);

    if (sidebar_id && isNaN(sidebar_id)) {
      return sendResponse(res, 400, "Invalid sidebar id");
    }

    if (sidebar_id) {
      const data = await newUploadToDDRM(
        req?.files?.upload_document,
        sidebar_id,
        req
      );
      actualPath = data.uploadedPath;
      parentId = data.parentId;
      req.body.sidebar_id = sidebar_id;
    }

    // const fileName = req?.files?.upload_document?.name;
    let fileName = "";
    if (req?.files?.upload_document?.name) {
      const fileExtension = path.extname(req.files.upload_document.name);
      fileName = `${Date.now()}${fileExtension}`;
    }

    const fileSizeInBytes = req?.files?.upload_document?.size;
    // Check if quota exceeds for the client

    try {
      const quotaResponse = await axios.get(
        `${process.env.SUPER_ADMIN_BACKEND_URL}/api/storage-type/get-quota/${process.env.CLIENT_INTERNAL_ID}`,
        {
          params: {
            fileSize: fileSizeInBytes, // Query param: fileSize=...
          },
        }
      );

      const quotaCheck = quotaResponse.data;
      if (!quotaCheck.data.allow) {
        return sendResponse(res, 400, quotaCheck.data.message);
      }
    } catch (error) {
      console.error(chalk.red(`Error checking quota: ${error.message}`));
      // return sendResponse(
      //   res,
      //   500,
      //   "Failed to verify storage quota. Please try again later."
      // );
    }

    if (!fs.existsSync(actualPath)) {
      fs.mkdirSync(actualPath, { recursive: true }); // Only create folder, not the file
      console.log(chalk.yellow(`Created folder: ${actualPath}`));
    }
    // Now move the file to the correct location (not to a folder named after document)
    actualPath = path.join(actualPath, fileName);
    await file.mv(actualPath); // move the file to the actual path with document name
    console.log(
      chalk.yellow(`File moved to: ${path.join(actualPath, fileName)}`)
    );
  }

  // Check if using third-party storage (AWS, Azure, or Google Cloud)
  let filePath = "";
  if (req.files) {
    if (process.env.STORAGE_TYPE === "third_party") {
      let uploadFilePath = await uploadFile("cloud", req?.files?.upload_document);
      uploadFilePath = path.join(process.cwd(), "public", uploadFilePath);

      const file = req?.files?.upload_document;
      if (process.env.PROVIDER === "AWS") {
        // console.log("filePath: ", uploadFilePath);
        filePath = await uploadToS3(file);
      } else if (process.env.PROVIDER === "Azure") {
        filePath = await uploadToAzure(
          uploadFilePath,
          req?.files?.upload_document.name
        );
      } else if (process.env.PROVIDER === "Google Cloud") {
        filePath = await uploadToGCP(
          uploadFilePath,
          req?.files?.upload_document.name
        );
      }

      // fs.unlinkSync(uploadFilePath);
    } else {
      const lastPublicIndex = actualPath.lastIndexOf("public");
      filePath = actualPath.substring(lastPublicIndex + "public".length);
    }
  }

  // Store the file path in the request body for database insert/update
  filePath = filePath
    ? process.env.STORAGE_TYPE === "third_party"
      ? `${filePath}`
      : `${process.env.HARMONY_BACKEND_URL}${filePath}`
    : req.body.upload_document;
  req.body.upload_document = filePath;

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (!organization || organization?.length === 0) {
    return sendResponse(
      res,
      200,
      "Please select at least one business structure"
    );
  }

  const isLocked = sidebar_id ? 1 : 0;
  req.body = await encodeAndStringifyFields(req.body);
  req.body.is_locked = isLocked;

  const { query, values } = id
    ? updateQueryBuilder(DocumentCreation, req.body)
    : createQueryBuilder(DocumentCreation, req.body);

  const [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Document Creation",
    id ? id : result.insertId
  );
  let repoData = {
    name: document_name,
    url: filePath,
    parent_id: parentId && parentId !== "undefined" ? parentId : null,
    organization,
    type: "file",
    document_creation_id: id ? id : result.insertId,
    created_by: id ? result?.created_by : req.user.sessionid,
    updated_by: id ? req.user.sessionid : null,
    is_locked: isLocked,
    sidebar_id: sidebar_id ? sidebar_id : null,
  };
  if (ddrm_id && ddrm_id != "null") {
    repoData.id = ddrm_id;
  }
  let createdData;
  const { query: RepositoryQuery, values: RepositoryValues } = ddrm_id && ddrm_id != "null"
    ? updateQueryBuilder(Repository, repoData)
    : createQueryBuilder(Repository, repoData);
  [createdData] = await db.query(RepositoryQuery, RepositoryValues);
  console.log("createdData: ", createdData);

  const data = {
    organization: req?.body?.organization,
    department: req?.body?.department,
    repository_id: createdData?.insertId || ddrm_id,
    created_by: req.user.sessionid,
    document_name: document_name,
    action_type: id ? "edit" : "upload",
    document_status: req?.body?.document_status,
  };
  // audit log for document
  await auditLogFunction(data);

  let resultData = {
    ddrm_id: createdData?.insertId ? createdData?.insertId : ddrm_id,
    url: filePath,
  };
  // console.log("resultData: ", resultData);

  return sidebar_id
    ? sendResponse(res, 200, resultData, `Record ${status} Successfully`)
    : sendResponse(res, 200, `Record ${status} Successfully`);
};

export const getDocument = async (req, res) => {
  let { id } = req.params;
  const ddrm_id = req.query.ddrm_id;
  let isArchivedCondition = "AND document_creation.is_archived = 0";
  if (ddrm_id) {
    const [data] = await getRecord("repository", "id", ddrm_id);
    id = data?.document_creation_id;
    isArchivedCondition = "";
  }
  const condition = await whereCondition({
    table: "document_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  //if id is provided then add in audit log
  if (id) {
    const [documentData] = await getRecord(
      "repository",
      "document_creation_id",
      id
    );
    const [documentCreationData] = await getRecord(
      "document_creation",
      "id",
      id
    );
    const data = {
      organization: documentData?.organization,
      department: documentData?.department,
      repository_id: documentData?.id,
      created_by: req.user.sessionid,
      document_name: documentData?.name,
      action_type: "view",
      document_status: documentCreationData?.document_status,
    };
    // audit log for document
    await auditLogFunction(data);
  }

  const searchFields = [
    "document_creation.document_name",
    "document_creation.unique_id",
    "document_creation.description",
    "users.name",
    "organization.name",
    "document_creation.document_amendment",
    "document_creation.description",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = document_creation.created_by",
    },
    {
      type: "left",
      targetTable: "document_type",
      onCondition: "document_type.id = document_creation.document_type",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = document_creation.organization",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.document_creation_id = document_creation.id",
    },
    {
      type: "left",
      targetTable: "users as owner",
      onCondition: "owner.id = document_creation.owner_user_id",
    },
    {
      type: "left",
      targetTable: "roles as owner_role",
      onCondition: "owner_role.id = users.role",
    },
    {
      type: "left",
      targetTable: "department as owner_department",
      onCondition: "owner_department.id = users.department",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const viewingRightCondition = req.user.isSuperAdmin
    ? ""
    : `AND (
    (JSON_VALID(document_creation.viewing_rights_employees) 
     AND JSON_CONTAINS(document_creation.viewing_rights_employees, '${req.user.sessionid}'))
    OR document_creation.created_by = '${req.user.sessionid}' OR document_creation.document_classification IN ('Restricted', 'Unclassified'))`;

  const fetchQuery = `SELECT document_creation.*, repository.id as ddrm_id, repository.parent_id, CONCAT(users.name, ' ', users.surname) AS created_by,
  document_type.created_by AS created_by_id,
  users.profile as created_by_profile,
  CONCAT(owner.name, ' ', owner.surname) AS owner_name,
  owner_role.name AS owner_role,
  owner_department.name AS owner_department,
  document_type.name AS document_type_name, organization.name AS organization_name FROM document_creation ${joinsRecord} WHERE document_creation.deleted = 0 ${isArchivedCondition} ${viewingRightCondition} ${searchCondition} ${condition}`;
  let [records] = await db.query(fetchQuery);

  records = await decodeAndParseFields(records);
  for (const record of records) {
    record.description = await decodeSingle_statement(record.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM document_creation WHERE deleted = 0 AND name = ?`,
        [record.name]
      );
      const arr = organizations.map((item) => item.organization);
      record.organizations = arr;
    }
    record.viewing_rights_details = await getUserListByIds(
      record?.viewing_rights_employees
    );
    record.modification_rights_details = await getUserListByIds(
      record?.modification_rights_employees
    );
    record.deleting_rights_details = await getUserListByIds(
      record?.deleting_rights_employees
    );
  }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, records, totalRecord);
};

export const deleteDocumentCreation = async (req, res) => {
  const { id } = req.params;

  const [checkRecord] = await db.query(
    `SELECT * FROM document_creation WHERE id = ? AND deleted = 0`,
    [id]
  );
  if (checkRecord.length === 0) {
    return sendResponse(res, 404, "Record not found");
  }

  if (checkRecord[0]?.is_locked === 1)
    return sendResponse(
      res,
      400,
      `cannot delete '${checkRecord[0]?.name}': Permission denied`
    );

  // also delete from repository
  const deleteFromRepoRecord = await db.query(
    `UPDATE repository SET deleted = 1 WHERE document_creation_id = ?`,
    [id]
  );
  const deleteRecord = await db.query(
    `UPDATE document_creation SET deleted = 1 WHERE id = ?`,
    [id]
  );
  if (deleteRecord) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "DocumentCreation",
      id
    );

    const { organization, department, document_name } = checkRecord[0];
    const data = {
      organization: organization,
      department: department,
      repository_id: id,
      created_by: req.user.sessionid,
      document_name: document_name,
      action_type: "delete",
      document_status: checkRecord[0]?.document_status,
    };
    await auditLog(data);

    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};

export const getDocumentOverviewData = async (req, res) => {
  const countQuery = `
  SELECT 
    COUNT(repository.id) AS total_documents,
    SUM(CASE WHEN document_creation.expiry_date < CURDATE() THEN 1 ELSE 0 END) AS expired_documents,
    SUM(CASE WHEN repository.is_archived = 1 THEN 1 ELSE 0 END) AS archived_documents
  FROM repository 
  LEFT JOIN document_creation 
    ON repository.document_creation_id = document_creation.id
  WHERE repository.deleted = 0 
    AND repository.type = "file";
`;

  // Execute query
  const [result] = await db.query(countQuery);
  // Send response
  return sendResponse(res, 200, result[0]);
};

export const shareDocument = async (req, res) => {
  const { id, employees } = req.body;
  const [documentRecord] = await getRecord("repository", "id", id);
  const docUrl = documentRecord?.url;
  for (let i = 0; i < employees.length; i++) {
    await insertNotification(
      "Document shared",
      docUrl,
      employees[i],
      "url",
      req.user.sessionid
    );
  }
  return sendResponse(res, 200, "document shared successfully");
};

export const documentDownloadLog = async (req, res) => {
  const { id } = req.body;
  const [repositoryRecord] = await getRecord("repository", "id", id);
  const [documentRecord] = await getRecord(
    "document_creation",
    "id",
    repositoryRecord?.document_creation_id
  );
  const data = {
    organization: repositoryRecord?.organization,
    department: repositoryRecord?.department,
    repository_id: repositoryRecord?.id,
    created_by: req.user.sessionid,
    document_name: repositoryRecord?.name,
    action_type: "download",
    document_status: documentRecord?.document_status,
  };
  await auditLogFunction(data);
  return sendResponse(res, 200, {url: repositoryRecord.url}, "File downloaded successfully");
};

export const getExpiringDocument = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "document_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  //if id is provided then add in audit log
  if (id) {
    const [documentData] = await getRecord(
      "repository",
      "document_creation_id",
      id
    );
    const data = {
      organization: documentData?.organization,
      department: documentData?.department,
      repository_id: documentData.id,
      created_by: req.user.sessionid,
      document_name: documentData.name,
      action_type: "view",
      document_status: req?.body?.document_status,
    };
    // audit log for document
    await auditLogFunction(data);
  }

  const searchFields = [
    "document_creation.document_name",
    "document_creation.unique_id",
    "document_creation.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = document_creation.created_by",
    },
    {
      type: "left",
      targetTable: "document_type",
      onCondition: "document_type.id = document_creation.document_type",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = document_creation.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const fetchQuery = `SELECT document_creation.*, CONCAT(users.name, ' ', users.surname) AS created_by,
  document_type.created_by AS created_by_id,
  users.profile as created_by_profile,
  document_type.name AS document_type_name, organization.name AS organization_name FROM document_creation ${joinsRecord} WHERE document_creation.deleted = 0 AND document_creation.expiry_date < CURDATE() ${searchCondition} ${condition}`;
  let [records] = await db.query(fetchQuery);
  records = await decodeAndParseFields(records);
  for (const record of records) {
    // get retention policy if exists
    if (record.document_type) {
      const [retentionPolicy] = await getRecord(
        "retention_policy_setting",
        "document_type_id",
        record.document_type
      );
      if (retentionPolicy) {
        record.retention_policy = retentionPolicy.retention_period;
      }
    }
    record.description = await decodeSingle_statement(record.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM document_creation WHERE deleted = 0 AND name = ?`,
        [record.name]
      );
      const arr = organizations.map((item) => item.organization);
      record.organizations = arr;
    }
    record.viewing_rights_details = await getUserListByIds(
      record?.viewing_rights_employees
    );
    record.modification_rights_details = await getUserListByIds(
      record?.modification_rights_employees
    );
    record.deleting_rights_details = await getUserListByIds(
      record?.deleting_rights_employees
    );
  }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, records, totalRecord);
};

export const updateDocumentStatus = async (req, res) => {
  const { status, id } = req.body;
  const updateQuery = `UPDATE document_creation set status = ? where id = ?`;
  const [update] = await db.query(updateQuery, [status, id]);
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export function scheduleBackup(backupType) {
  let cronExpression = "";

  switch (backupType) {
    case "daily":
      cronExpression = "0 0 * * *"; // Every day at midnight
      break;
    case "weekly":
      cronExpression = "0 0 * * 0"; // Every Sunday at midnight
      break;
    case "monthly":
      cronExpression = "0 0 1 * *"; // Every 1st of the month at midnight
      break;
    case "yearly":
      cronExpression = "0 0 1 1 *"; // Every January 1st at midnight
      break;
    default:
      console.log("Invalid backup type.");
      return;
  }

  cron.schedule(cronExpression, async () => {
    console.log(`Starting ${backupType} backup...`);

    const publicDir = path.join(process.cwd(), "public");
    const parentDir = path.join(publicDir, "backup");

    if (!fs.existsSync(parentDir)) {
      fs.mkdirSync(parentDir, { recursive: true });
    }

    const todayDate = moment().format("YYYY-MM-DD_HH-mm"); // cannot use colon (:) in folder name, reserved in windows

    const backupDir = path.join(parentDir, `${type}_${todayDate}`);
    fs.mkdirSync(backupDir, { recursive: true });

    // source folder to backup
    const sourceFolder = path.join(publicDir, "root");

    await copyRecursiveAsync(sourceFolder, backupDir);
  });

  console.log(
    `${backupType} backup scheduled with cron expression: ${cronExpression}`
  );
}

// backupDDRM("daily");

export const getDocumentWithVersions = async (req, res) => {
  const { id } = req.params;
  const page = parseInt(req.query.page) || 1;
  const pageSize = parseInt(req.query.pageSize) || 10;
  const offset = (page - 1) * pageSize;

  // const query = `WITH RECURSIVE linked_documents AS (
  //               --  Start with the current document
  //               SELECT *
  //               FROM document_creation
  //               WHERE id = ? AND deleted = 0

  //               UNION ALL

  //               -- Fetch documents linked to the current document
  //               SELECT dc.*
  //               FROM document_creation dc
  //               INNER JOIN linked_documents ld ON dc.select_revised_document = ld.id
  //               WHERE dc.deleted = 0
  //           )
  //           SELECT * FROM linked_documents ORDER BY id DESC  LIMIT ? OFFSET ?`;

  const query = `WITH RECURSIVE archived_documents AS (
                -- Start with the current document
                SELECT *
                FROM document_creation
                WHERE id = ? AND deleted = 0

                UNION ALL

                -- Fetch documents linked to the current document
                SELECT dc.*
                FROM document_creation dc
                INNER JOIN archived_documents ad ON ad.select_revised_document = dc.id
                WHERE dc.deleted = 0 AND dc.is_archived = 1
            )
            SELECT 
                ad.*, 
                CONCAT(u1.name, ' ', u1.surname) AS created_by_name, u1.profile AS created_by_profile,
                CONCAT(u2.name, ' ', u2.surname) AS updated_by_name, u2.profile AS updated_by_profile
            FROM archived_documents ad
            LEFT JOIN users u1 ON ad.created_by = u1.id
            LEFT JOIN users u2 ON ad.updated_by = u2.id
            ORDER BY ad.id DESC
            LIMIT ? OFFSET ?`;

  let [records] = await db.query(query, [id, pageSize, offset]);
  records = await decodeAndParseFields(records);

  for (let record of records) {
    record.viewing_rights_details = await getUserListByIds(
      record?.viewing_rights_employees
    );
    record.modification_rights_details = await getUserListByIds(
      record?.modification_rights_employees
    );
    record.deleting_rights_details = await getUserListByIds(
      record?.deleting_rights_employees
    );
  }

  const totalRecord = records.length;
  return sendResponse(res, 200, records, totalRecord);
};