import db from "../db-config.js";
import {
  checkNameInOrganization,
  countQueryCondition,
  createQueryBuilder,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  processesSingleDDRMDocument,
  searchConditionRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import DocumentRepository from "../sequelize/DocumentRepositorySchema.js";

/**Function to create and update Document Repository  */
export const createUpdatedDocumentRepository = async (req, res) => {
  const {
    id,
    document_name,
    document_description,
    document_path,
    organization,
    department,
    sidebar_id = 158,
  } = req.body;

  req.body.document_description = await encodeSingle_statement(
    document_description
  );
  // req.body.allowed_access_to = JSON.stringify(req.body.allowed_access_to);

  const document = req.files && req.files.document;

  req.body.document_path = "";
  if (document) {
    // req.body.document_path = await uploadFile("document_repository", document);
    const ddrm_id = await processesSingleDDRMDocument(
      "document_repository",
      sidebar_id,
      document,
      req
    );
    req.body.ddrm_id = ddrm_id;
  }

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**Check that this organization have that particular name or not*/
  const checkNameWithOrganization = await checkNameInOrganization(
    "document_repository",
    document_name,
    organizationId,
    "document_name"
  );

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    if (
      checkNameWithOrganization.length > 0 &&
      checkNameWithOrganization[0]?.id != id
    ) {
      return sendResponse(
        res,
        400,
        `Record with  Document name ${document_name} already exists.`
      );
    }

    const { query, values } = updateQueryBuilder(DocumentRepository, req.body);
    await db.query(query, values);
    /** update the file path if document is updated */
    // if (req.body.document_path) {
    //   const updateDocumentRepositoryQuery = `UPDATE document_repository
    //       SET document_path=? WHERE id=?`;
    //   await db.query(updateDocumentRepositoryQuery, [
    //     req.body.document_path,
    //     id,
    //   ]);
    // }

    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "update",
      "Document Repository",
      `This user Update other Skills which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    if (checkNameWithOrganization.length > 0) {
      return sendResponse(res, 400, "This record is already exists");
    }
    req.body.created_by = req.user.sessionid;
    /** Insert Document Repository details */
    const { query, values } = createQueryBuilder(DocumentRepository, req.body);
    await db.query(query, values);
    /** Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Document Repository",
      `This user created a new other Skills named ${document_name} for organization ${organizationId}`
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Document Repository */
export const viewAllDocumentRepository = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "document_repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "document_repository.document_name",
    "document_repository.document_description",
    "CONCAT(users.name , ' ' , users.surname)",
    "audit_scheduling.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = document_repository.created_by",
    },
    // {
    //   type: "left",
    //   targetTable: "users as u1",
    //   onCondition: "u1.id = document_repository.allowed_access_to",
    // },
    {
      type: "left",
      targetTable: "audit_scheduling",
      onCondition: "audit_scheduling.id = document_repository.associated_audit",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = document_repository.organization",
    },
    {
      type: "left",
      targetTable: "document_type",
      onCondition: "document_type.id = document_repository.document_type",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = document_repository.ddrm_id",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all Document Repository */
  const documentRepositoryDataFetchQuery = `SELECT document_repository.id,document_repository.created_by as created_by_id,document_repository.created_at,document_repository.document_type,document_repository.associated_audit,document_repository.document_name,document_repository.document_description,document_repository.allowed_access_to,
  repository.url as document_path,
    audit_scheduling.name as associated_audit_name,
    document_type.name as document_type_name,CONCAT(users.name , ' ' , users.surname ) as created_by ,users.profile as created_by_profile,organization.name as organization_name,document_repository.organization
    FROM document_repository  
    ${joinsRecord} 
    where document_repository.deleted = 0  ${searchCondition} ${condition}`;
  const [documentRepositoryDataFetch] = await db.query(
    documentRepositoryDataFetchQuery
  );

  for (const item of documentRepositoryDataFetch) {
    item.document_description = await decodeSingle_statement(
      item.document_description
    );
    item.allowed_access_to = JSON.parse(item.allowed_access_to);
  }

  for (const item of documentRepositoryDataFetch) {
    const users = item.allowed_access_to;
    const arr = [];
    if (users && Array.isArray(users)) {
      for (const user of users) {
        const [fetchUser] = await db.query(
          `SELECT CONCAT(name, ' ', surname) AS name, profile FROM users WHERE id = ${user}`
        );
        const obj = {};
        obj.id = user;
        (obj.name = fetchUser[0]?.name),
          (obj.profile = fetchUser[0]?.profile),
          arr.push(obj);
        item.allowed_access_to_name = arr;
      }
    }
  }

  /**Count all Document Repository */
  const totalRecord = await countQueryCondition(
    documentRepositoryDataFetchQuery
  );

  return sendResponse(res, 200, documentRepositoryDataFetch, totalRecord);
};

/**Function to delete a specific Document Repository */
export const deleteDocumentRepository = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("document_repository", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Document Repository", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
