import {
  countQueryCondition,
  decodeAndParseFields,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import db from "../db-config.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import InjuryOnDuty from "../sequelize/InjuryOnDutyFormSchema.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";

/**Function to create Safety Incident Form  */
export const createInjuryOnDutyForm = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const { id, sidebar_id = 176 } = req.body;
  const sessionId = req.user.sessionid;
  let organizationId = req.body.organization;
  const { department, comments } = req.body;

  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0]?.organization;
    if (organizationId) {
      req.body.organization = organizationId;
    }
  }
  req.body.notification = req.body?.notification?.map((user) => user.user_id);
  req.body.comments = await encodeSingle_statement(comments);

  /** Multiple Image Handle */
  const injured_persons = req.body?.injured_persons || [];
  for (let i = 0; i <= injured_persons.length; i++) {
    if (
      (req.files && req.files[`injured_persons[${i}][alcohol_file]`]) ||
      req.body[`injured_persons[${i}][alcohol_file]`]
    ) {
      const file = req.files && req.files[`injured_persons[${i}][alcohol_file]`];
      if (typeof file != "string" && typeof file == "object" && file !== null) {
        // const filePath = await uploadFile("vehicle_incident_form", file);
        const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
        // req.body[`injured_persons`][i]["file"] = filePath;
        req.body[`injured_persons`][i]["alcohol_ddrm_id"] = ddrm_id;
      }
    }
    if (
      (req.files && req.files[`injured_persons[${i}][substance_file]`]) ||
      req.body[`injured_persons[${i}][substance_file]`]
    ) {
      const file = req.files && req.files[`injured_persons[${i}][substance_file]`];
      if (typeof file != "string" && typeof file == "object" && file !== null) {
        // const filePath = await uploadFile("vehicle_incident_form", file);
        const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
        // req.body[`injured_persons`][i]["file"] = filePath;
        req.body[`injured_persons`][i]["substance_ddrm_id"] = ddrm_id;
      }
    }
  }

  req.body.created_by = sessionId;
  const { query, values } = id
    ? updateQueryBuilder(InjuryOnDuty, req.body)
    : createQueryBuilder(InjuryOnDuty, req.body);
  const [createInjuryOnDutyForm] = await db.query(query, values);
  if (!id && createInjuryOnDutyForm.insertId === 0) {
    return sendResponse(res, 404, "Cannot create record");
  }

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Injury On duty Form",
    `This user created a new Injury On Duty Form Record for organization ${organizationId}`
  );

  return sendResponse(res, 200, "Record created successfully");
};

export const getInjuryOnDutyForm = async (req, res) => {
  const { type } = req.query;
  const { id } = req.params;

  const condition = await whereCondition({
    table: "injury_on_duty",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
    columnName: type,
  });
  const searchTableName = [
    "injury_on_duty.id", // Primary key in InjuryOnDuty schema
    "injury_on_duty.injured_persons", // List of injured persons
    "injury_on_duty.id_number", // ID number (if this exists, needs to be added to schema)
    "injury_on_duty.employee_type", // Employee type (if this exists, needs to be added to schema)
    "injury_on_duty.state_name", // State name (if this exists, needs to be added to schema)
    "injury_on_duty.alcohol_substance_test_status", // Status of substance test
    "injury_on_duty.test_result", // Test result
    "injury_on_duty.injury_severity_level", // Injury severity level
    "injury_on_duty.hospital_name", // Hospital name
    "injury_on_duty.doctor_name", // Doctor's name
    "injury_on_duty.practice_number", // Doctor's practice number
    "injury_on_duty.contact_details", // Contact details
    "injury_on_duty.type_of_injury", // Injury type (if this exists, needs to be added to schema)
    "injury_on_duty.parts_of_body_affected", // Affected body parts (if this exists, needs to be added to schema)
    "injury_on_duty.type_of_ppe_worn", // PPE worn
    "injury_on_duty.reportable_cases_classification", // Classification of reportable cases
    "injury_on_duty.first_aid_applied", // First aid applied (if this exists, needs to be added to schema)
    "injury_on_duty.comments", // Comments (if this exists, needs to be added to schema)
    "injury_on_duty.form_completed_by", // Person who completed the form
    "injury_on_duty.designation", // Designation (if this exists, needs to be added to schema)
    "organization.name", // Organization name
    "department.name", // Department name
    "CONCAT(user.name , ' ' , user.surname)", // User name concatenation (assuming User model has name and surname fields)
    "roles.name", // Role name (if associated in the schema)
  ];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = injury_on_duty.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = injury_on_duty.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = injury_on_duty.department",
    },
    {
      type: "left",
      targetTable: "incident",
      onCondition: "incident.id = injury_on_duty.incident_id",
    },
  ];

  const joinCondition = await makeJoins(joins);
  const injuryOnDutyFormQuery = `SELECT injury_on_duty.*, incident.incident_title as incident_name, incident.incident_actual_date as date_of_incident, incident.incident_actual_time as time_of_incident, incident.description as incident_description,  organization.name as organization_name ,department.name as department_name , users.name as created_by_name , users.surname as created_by_surname  , users.profile as created_by_profile  FROM injury_on_duty ${joinCondition} WHERE injury_on_duty.deleted = 0 ${condition} ${searchCondition}`;
  const totalRecord = await countQueryCondition(injuryOnDutyFormQuery);

  let [injuryOnDutyForm] = await db.query(injuryOnDutyFormQuery);
  injuryOnDutyForm = await decodeAndParseFields(injuryOnDutyForm);
  for (const data of injuryOnDutyForm) {
    const { injured_persons, notification } = data;

    for (let i = 0; i < injured_persons.length; i++) {
      if (injured_persons[i].alcohol_ddrm_id) {
        const [repositoryRecordAlcohol] = await getRecord("repository", "id", injured_persons[i]?.alcohol_ddrm_id);
        const [repositoryRecordSubstance] = await getRecord("repository", "id", injured_persons[i]?.substance_ddrm_id);
        injured_persons[i].alcohol_file = repositoryRecordAlcohol?.url;
        injured_persons[i].substance_file = repositoryRecordSubstance?.url;
      }
    }

    // if (data.tested_by == "Internal") {
    //   if (data.tested_by_employee) {
    //     const [userFetch] = await db.query(
    //       `SELECT CONCAT(name , ' '  , surname) AS name , profile , roles.name as role_name , department.name as department_name FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON users.department = department.id WHERE id = ${Number(
    //         data.tested_by_employee
    //       )}`
    //     );
    //     data.tested_by_employee_name = userFetch[0]?.name;
    //     data.tested_by_employee_profile = userFetch[0]?.profile;
    //   }
    // }
    /** Notification  User*/
    if (notification && notification.length > 0 && notification[0] != "") {
      let [notificationData] = await db.query(
        `SELECT users.id as user_id, CONCAT(users.name , ' ' , users.surname) as name,  roles.name as role_name , department.name as department 
         FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON users.department = department.id WHERE users.id IN (${notification})`
      );
      data.notification = notificationData;
    }
  }
  return sendResponse(res, 200, injuryOnDutyForm, totalRecord);
};

export const deleteInjuryOnDutyForm = async (req, res) => {
  const { id } = req.params;
  const deleteForm = await deleteRecord("injury_on_duty", id);
  if (deleteForm) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Injury On Duty Form",
      `This user deleted a new Injury On duty Form Record for organization ${id}`
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Cannot delete record");
};
