import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  whereCondition,
  countQueryCondition,
  deleteSettingRecord,
  encodeSingle_statement,
  decodeSingle_statement,
  settingsUpdater,
  uniqueIdGenerator,
} from "../helper/general.js";
import WorkLocation from "../sequelize/WorkLocationSchema.js";

/** Function to create or update WorkLocation */
export const createUpdateWorkLocation = async (req, res) => {
  const { id, name, description } = req.body;
  const encodedDescription = await encodeSingle_statement(description);

  if (id) {
    const result = await settingsUpdater(
      WorkLocation,
      req.body.organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 404, "No data found");
    }
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "WorkLocation",
      `This user updated WorkLocation with id: ${id}`
    );
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const insertPromises = req.body.organization.map(async (org) => {
      const [checkName] = await db.query(
        `SELECT name FROM work_location WHERE name = ? AND deleted = 0 AND organization = ?`,
        [name, org]
      );
      if (checkName.length === 0) {
        const unique_id = await uniqueIdGenerator(
          org,
          req.body?.department || null,
          "WorkLocation",
          "work_location",
          "unique_id",
          "unique_id"
        );

        await db.query(
          `INSERT INTO work_location (name, description, created_by, organization,unique_id) VALUES (?, ?, ?, ?,?)`,
          [name, encodedDescription, req.user.sessionid, org,unique_id]
        );
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "WorkLocation",
          `This user created a new WorkLocation '${name}' for organization ${org}`
        );
      }
    });
    await Promise.all(insertPromises);
    return sendResponse(res, 200, "Record created successfully");
  }
};

/** Function to view all or single WorkLocation */
export const viewWorkLocation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "work_location",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchFields = [
    "work_location.name",
    "work_location.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = work_location.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = work_location.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const fetchQuery = `SELECT work_location.*, users.name AS created_by ,users.surname as created_by_surname, organization.name AS organization_name FROM work_location ${joinsRecord} WHERE work_location.deleted = 0 ${searchCondition} ${condition}`;
  let [records] = await db.query(fetchQuery);

  for (const record of records) {
    record.description = await decodeSingle_statement(record.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM work_location WHERE deleted = 0 AND name = ?`,
        [record.name]
      );
      const arr = organizations.map((item) => item.organization);
      record.organizations = arr;
    }
  }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, records, totalRecord);
};

/** Function to delete WorkLocation */
export const deleteWorkLocation = async (req, res) => {
  const { id } = req.params;
  const deleteRecord = await deleteSettingRecord("work_location", id);
  if (deleteRecord) {
    await insertActivityLog(req.user.sessionid, "delete", "WorkLocation", id);
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};
