import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorAgreements,
  getContractorAgreements,
  deleteContractorAgreements,
  sendAgreements,
  fillForm,
} from "../controller/contractorAgreementsController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Agreements
 *   description: API operations for managing contractor questionnaires
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorAgreements:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor questionnaire
 *         contractor:
 *           type: integer
 *           description: ID of the contractor
 *         status:
 *           type: string
 *           description: The current status of the questionnaire
 *         form:
 *           type: string
 *           description: The form data in long text format
 *         public_link:
 *           type: string
 *           description: Public link for the questionnaire
 *         images:
 *           type: string
 *           description: Associated images in long text format
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         deleted:
 *           type: integer
 *           description: Whether the questionnaire is deleted (0 or 1)
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the questionnaire
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the questionnaire was created
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the questionnaire
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the questionnaire was last updated
 */

const contractorAgreementsRoutes = Router();

/**
 * @swagger
 * /api/contractorAgreements/create-update-contractor-agreements:
 *   post:
 *     summary: Create or update a contractor questionnaire
 *     tags: [Contractor Agreements]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorAgreements'
 *     responses:
 *       200:
 *         description: The contractor questionnaire was successfully created/updated
 */

contractorAgreementsRoutes.post(
  "/create-update-contractor-agreements",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      259,
      "contractor_agreements"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorAgreements)
);

/**
 * @swagger
 * /api/contractorAgreements/get-contractor-agreements:
 *   get:
 *     summary: Get all contractor questionnaires
 *     tags: [Contractor Agreements]
 *     responses:
 *       200:
 *         description: List of all contractor questionnaires
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorAgreements'
 */

contractorAgreementsRoutes.get(
  "/get-contractor-agreements",
  verifyToken,
  sideBarPermissionCheck("view", 259, "contractor_agreements"),
  controllerWrapper(getContractorAgreements)
);

/**
 * @swagger
 * /api/contractorAgreements/get-contractor-agreements/{id}:
 *   get:
 *     summary: Get a contractor questionnaire by ID
 *     tags: [Contractor Agreements]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor questionnaire
 *     responses:
 *       200:
 *         description: Contractor questionnaire data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorAgreements'
 */

contractorAgreementsRoutes.get(
  "/get-contractor-agreements/:id",
  verifyToken,
  sideBarPermissionCheck("view", 259, "contractor_agreements"),
  controllerWrapper(getContractorAgreements)
);

/**
 * @swagger
 * /api/contractorAgreements/delete-contractor-agreements/{id}:
 *   delete:
 *     summary: Delete a contractor questionnaire by ID
 *     tags: [Contractor Agreements]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor questionnaire to delete
 *     responses:
 *       200:
 *         description: Contractor questionnaire was deleted
 */

contractorAgreementsRoutes.delete(
  "/delete-contractor-agreements/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 259, "contractor_agreements"),
  controllerWrapper(deleteContractorAgreements)
);

/**
 * @swagger
 * /api/contractorAgreements/send-agreements:
 *   get:
 *     summary: Send a contractor questionnaire
 *     tags: [Contractor Agreements]
 *     responses:
 *       200:
 *         description: The questionnaire was successfully sent
 */

contractorAgreementsRoutes.post(
  "/send-agreements",
  verifyToken,
  controllerWrapper(sendAgreements)
);

/**
 * @swagger
 * /api/contractorAgreements/fill-agreements:
 *   post:
 *     summary: Fill out a contractor questionnaire form
 *     tags: [Contractor Agreements]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorAgreements'
 *     responses:
 *       200:
 *         description: The form was successfully submitted
 */

contractorAgreementsRoutes.post(
  "/fill-agreements",
  controllerWrapper(fillForm)
);

export default contractorAgreementsRoutes;
