// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorAuditProtocol,
  deleteContractorAuditProtocol,
  saveAuditRecord,
  viewAllContractorAuditProtocol,
  viewAllAuditRecord,
} from "../controller/contractorAuditProtocolController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import {
  //   contractorAuditProtocolCreateUpdateValidationSchema,
  auditTemplateCreateUpdateValidationSchema,
} from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Audit Protocol
 *   description: API operations for managing all Contractor Audit Protocol
 */
const contractorAuditProtocolRoutes = Router();

/**
 * @swagger
 * /api/contractorAuditProtocol/create-update-contractor-audit-protocol:
 *   post:
 *     summary: Create or update an Contractor Audit Protocol
 *     tags: [Contractor Audit Protocol]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: { id: "id", name: "name", description: "description", organization: [] }
 */
contractorAuditProtocolRoutes.post(
  "/create-update-contractor-audit-protocol",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      274,
      "contractor_audit_protocol"
    )(req, res, next);
  },
  //   validateRequest(contractorAuditProtocolCreateUpdateValidationSchema),
  controllerWrapper(createUpdateContractorAuditProtocol)
);

/**
 * @swagger
 * /api/contractorAuditProtocol/get-contractor-audit-protocol:
 *   get:
 *     summary: Get a list of all Contractor Audit Protocol
 *     tags: [Contractor Audit Protocol]
 */
contractorAuditProtocolRoutes.get(
  "/get-contractor-audit-protocol",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_protocol"),
  controllerWrapper(viewAllContractorAuditProtocol)
);

/**
 * @swagger
 * /api/contractorAuditProtocol/get-contractor-audit-protocol/{id}:
 *   get:
 *     summary: Get a specific record of Contractor Audit Protocol
 *     tags: [Contractor Audit Protocol]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit execution to retrieve
 *         schema:
 *           type: string
 */
contractorAuditProtocolRoutes.get(
  "/get-contractor-audit-protocol/:id",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_protocol"),
  controllerWrapper(viewAllContractorAuditProtocol)
);

/**
 * @swagger
 * /api/contractorAuditProtocol/delete-contractor-audit-protocol/{id}:
 *   delete:
 *     summary: Delete a specific record of Contractor Audit Protocol
 *     tags: [Contractor Audit Protocol]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit execution to delete
 *         schema:
 *           type: string
 */
contractorAuditProtocolRoutes.delete(
  "/delete-contractor-audit-protocol/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 274, "contractor_audit_protocol"),
  controllerWrapper(deleteContractorAuditProtocol)
);

/**
 * @swagger
 * /api/contractorAuditProtocol/save-audit-record:
 *   post:
 *     summary: Save a specific audit record using audit template
 *     tags: [Contractor Audit Protocol]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *               questions:
 *                 type: array
 *                 items:
 *                   type: string
 *               contractor_audit_protocol:
 *                 type: string
 *           example: { name: "name", description: "Description", organization: [], department: "department", questions: [], contractor_audit_protocol: "contractor_audit_protocol" }
 */
contractorAuditProtocolRoutes.post(
  "/save-audit-record",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_protocol"),
  // validateRequest(auditTemplateCreateUpdateValidationSchema),
  controllerWrapper(saveAuditRecord)
);

/**
 * @swagger
 * /api/contractorAuditProtocol/get-audit-record:
 *   get:
 *     summary: Get a list of all Audit Records
 *     tags: [Contractor Audit Protocol]
 */
contractorAuditProtocolRoutes.get(
  "/get-audit-record",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_protocol"),
  controllerWrapper(viewAllAuditRecord)
);

/**
 * @swagger
 * /api/contractorAuditProtocol/get-audit-record/{id}:
 *   get:
 *     summary: Get a specific record of Audit Record
 *     tags: [Contractor Audit Protocol]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit record to retrieve
 *         schema:
 *           type: string
 */
contractorAuditProtocolRoutes.get(
  "/get-audit-record/:id",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_protocol"),
  controllerWrapper(viewAllAuditRecord)
);

export default contractorAuditProtocolRoutes;
