// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
// import {
//   createUpdateContractorAuditScheduling,
//   deleteContractorAuditScheduling,
//   viewAllContractorAuditScheduling,
// } from "../controller/contractorAuditSchedulingController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import {
  addContractorEvaluationTool,
  createUpdateContractorAuditScheduling,
  deleteContractorAuditScheduling,
  saveAuditRecord,
  uploadContractorReport,
  viewAllContractorAuditScheduling,
} from "../controller/contractorAuditSchedulingController.js";
import { getSpecificRecords } from "../controller/auditSchedulingController.js";
// import { contractorAuditSchedulingCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Audit Scheduling
 *   description: API operations for managing all Contractor Audit Scheduling
 */
const contractorAuditSchedulingRoutes = Router();

/**
 * @swagger
 * /api/contractorAuditScheduling/create-update-contractor-audit-scheduling:
 *   post:
 *     summary: Create or update a Contractor Audit Scheduling
 *     tags: [Contractor Audit Scheduling]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               contractor:
 *                 type: integer
 *               project:
 *                 type: integer
 *               type:
 *                 type: integer
 *               schedule_type:
 *                 type: string
 *                 enum: ["audit", "inspection", "monitoring", "assessment", "surveillance", "certifications", "testing and examination"]
 *               linked_to:
 *                 type: string
 *                 enum: ["Permits", "SOPs", "Policy", "Action", "Incident", "NCR", "Business Continuity"]
 *               planned_start_date:
 *                 type: string
 *                 format: date
 *               planned_end_date:
 *                 type: string
 *                 format: date
 *               actual_start_date:
 *                 type: string
 *                 format: date
 *               actual_end_date:
 *                 type: string
 *                 format: date
 *               source:
 *                 type: string
 *                 enum: ["Internal", "External"]
 *               lead_person:
 *                 type: integer
 *               reference_number:
 *                 type: integer
 *               participant_type:
 *                 type: string
 *                 enum: ["Internal", "External"]
 *               stakeholder:
 *                 type: string
 *               record_name:
 *                 type: integer
 *               location:
 *                 type: string
 *               repeat_when:
 *                 type: string
 *               cycles:
 *                 type: string
 *               members:
 *                 type: string
 *               owner:
 *                 type: string
 *               participants:
 *                 type: string
 *               scope:
 *                 type: string
 *               critical_business:
 *                 type: string
 *               financial_resource:
 *                 type: string
 *               value:
 *                 type: integer
 *               currency:
 *                 type: integer
 *               opening_meeting_agenda:
 *                 type: string
 *                 format: date
 *               closing_meeting_agenda:
 *                 type: string
 *                 format: date
 *               evaluation_tool_status:
 *                 type: string
 *                 enum: ["Applicable", "Not Applicable", "Added", "Not Added"]
 *               technological_resources:
 *                 type: string
 *               evaluation_tool:
 *                 type: integer
 *               status:
 *                 type: string
 *                 enum: ["Todo", "In progress", "Executed", "Overdue"]
 *               report:
 *                 type: string
 *           example: {
 *             contractor: 1,
 *             project: 2,
 *             type: 3,
 *             schedule_type: "audit",
 *             linked_to: "Policy",
 *             planned_start_date: "2024-01-01",
 *             planned_end_date: "2024-01-02",
 *             actual_start_date: "2024-01-03",
 *             actual_end_date: "2024-01-04",
 *             source: "Internal",
 *             lead_person: 4,
 *             reference_number: 12345,
 *             participant_type: "External",
 *             stakeholder: "Stakeholder Name",
 *             record_name: 5,
 *             location: "Location Details",
 *             repeat_when: "Monthly",
 *             cycles: "2",
 *             members: "Member Details",
 *             owner: "Owner Name",
 *             participants: "Participant Details",
 *             scope: "Audit Scope",
 *             critical_business: "Critical Process",
 *             financial_resource: "Budget Allocation",
 *             value: 10000,
 *             currency: 1,
 *             opening_meeting_agenda: "2024-01-05",
 *             closing_meeting_agenda: "2024-01-06",
 *             evaluation_tool_status: "Applicable",
 *             technological_resources: "Resource Details",
 *             evaluation_tool: 6,
 *             status: "In progress",
 *             report: "Report Content"
 *           }
 */

contractorAuditSchedulingRoutes.post(
  "/create-update-contractor-audit-scheduling",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      274,
      "contractor_audit_scheduling"
    )(req, res, next);
  },
  //   validateRequest(contractorAuditSchedulingCreateUpdateValidationSchema),
  controllerWrapper(createUpdateContractorAuditScheduling)
);

/**
 * @swagger
 * /api/contractorAuditScheduling/get-contractor-audit-scheduling:
 *   get:
 *     summary: Retrieve a list of all Contractor Audit Scheduling records
 *     tags: [Contractor Audit Scheduling]
 *     responses:
 *       200:
 *         description: A list of Contractor Audit Scheduling records
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   contractor:
 *                     type: integer
 *                   project:
 *                     type: integer
 *                   type:
 *                     type: integer
 *                   schedule_type:
 *                     type: string
 *                     enum: ["audit", "inspection", "monitoring", "assessment", "surveillance", "certifications", "testing and examination"]
 *                   linked_to:
 *                     type: string
 *                     enum: ["Permits", "SOPs", "Policy", "Action", "Incident", "NCR", "Business Continuity"]
 *                   planned_start_date:
 *                     type: string
 *                     format: date
 *                   planned_end_date:
 *                     type: string
 *                     format: date
 *                   actual_start_date:
 *                     type: string
 *                     format: date
 *                   actual_end_date:
 *                     type: string
 *                     format: date
 *                   source:
 *                     type: string
 *                     enum: ["Internal", "External"]
 *                   status:
 *                     type: string
 *                     enum: ["Todo", "In progress", "Executed", "Overdue"]
 *                 example: {
 *                   id: 1,
 *                   contractor: 2,
 *                   project: 3,
 *                   type: 4,
 *                   schedule_type: "audit",
 *                   linked_to: "Policy",
 *                   planned_start_date: "2024-01-01",
 *                   planned_end_date: "2024-01-02",
 *                   actual_start_date: "2024-01-03",
 *                   actual_end_date: "2024-01-04",
 *                   source: "Internal",
 *                   status: "In progress"
 *                 }
 */

contractorAuditSchedulingRoutes.get(
  "/get-contractor-audit-scheduling",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_scheduling"),
  controllerWrapper(viewAllContractorAuditScheduling)
);

/**
 * @swagger
 * /api/contractorAuditScheduling/get-contractor-audit-scheduling/{id}:
 *   get:
 *     summary: Get a specific record of Contractor Audit Scheduling
 *     tags: [Contractor Audit Scheduling]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit scheduling to retrieve
 *         schema:
 *           type: string
 */
contractorAuditSchedulingRoutes.get(
  "/get-contractor-audit-scheduling/:id",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_scheduling"),
  controllerWrapper(viewAllContractorAuditScheduling)
);
/**
 * @swagger
 * /api/contractorAuditScheduling/get-specific-records:
 *   get:
 *     summary: Get a specific record of Contractor Audit Scheduling
 *     tags: [Contractor Audit Scheduling]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit scheduling to retrieve
 *         schema:
 *           type: string
 */
contractorAuditSchedulingRoutes.get(
  "/get-specific-records",
  verifyToken,
  controllerWrapper(getSpecificRecords)
);

/**
 * @swagger
 * /api/contractorAuditScheduling/delete-contractor-audit-scheduling/{id}:
 *   delete:
 *     summary: Delete a specific record of Contractor Audit Scheduling
 *     tags: [Contractor Audit Scheduling]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit scheduling to delete
 *         schema:
 *           type: string
 */
contractorAuditSchedulingRoutes.delete(
  "/delete-contractor-audit-scheduling/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 274, "contractor_audit_scheduling"),
  controllerWrapper(deleteContractorAuditScheduling)
);

/**
 * @swagger
 * /api/contractorAuditScheduling/save-audit-record:
 *   post:
 *     summary: Save a specific contractor audit record using audit template
 *     tags: [Contractor Audit Execution]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *               questions:
 *                 type: array
 *                 items:
 *                   type: string
 *               audit_execution:
 *                 type: string
 *           example: { name: "name", description: "Description", organization: [], department: "department", questions: [], audit_execution: "audit_execution" }
 */
contractorAuditSchedulingRoutes.post(
  "/save-audit-record",
  verifyToken,
  sideBarPermissionCheck("view", 274, "contractor_audit_scheduling"),
  // validateRequest(auditTemplateCreateUpdateValidationSchema),
  controllerWrapper(saveAuditRecord)
);

/**
 * @swagger
 * /api/contractorAuditScheduling/upload-contractor-report:
 *   post:
 *     summary: upload a report for Contractor Audit Scheduling
 *     tags: [Contractor Audit Scheduling]
 */
contractorAuditSchedulingRoutes.post(
  "/upload-contractor-report",
  verifyToken,
  controllerWrapper(uploadContractorReport)
);
contractorAuditSchedulingRoutes.post(
  "/add-evaluation-tool",
  verifyToken,
  controllerWrapper(addContractorEvaluationTool)
);

export default contractorAuditSchedulingRoutes;
