import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorPreSite,
  getContractorPreSite,
  deleteContractorPreSite,
  sendPreSite,
  fillForm,
} from "../controller/contractorPreSiteController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor PreSite
 *   description: API operations for managing contractor pre-sites
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorPreSite:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor pre-site
 *         contractor:
 *           type: integer
 *           description: ID of the contractor
 *         status:
 *           type: string
 *           description: The current status of the pre-site
 *         form:
 *           type: string
 *           description: The form data in long text format
 *         public_link:
 *           type: string
 *           description: Public link for the pre-site
 *         images:
 *           type: string
 *           description: Associated images in long text format
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         deleted:
 *           type: integer
 *           description: Whether the pre-site is deleted (0 or 1)
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the pre-site
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the pre-site was created
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the pre-site
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the pre-site was last updated
 */

const contractorPreSiteRoutes = Router();

/**
 * @swagger
 * /api/contractorPreSite/create-update-contractor-pre-site:
 *   post:
 *     summary: Create or update a contractor pre-site
 *     tags: [Contractor PreSite]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorPreSite'
 *     responses:
 *       200:
 *         description: The contractor pre-site was successfully created/updated
 */

contractorPreSiteRoutes.post(
  "/create-update-contractor-pre-site",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      268,
      "contractor_pre_site"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorPreSite)
);

/**
 * @swagger
 * /api/contractorPreSite/get-contractor-pre-site:
 *   get:
 *     summary: Get all contractor pre-sites
 *     tags: [Contractor PreSite]
 *     responses:
 *       200:
 *         description: List of all contractor pre-sites
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorPreSite'
 */

contractorPreSiteRoutes.get(
  "/get-contractor-pre-site",
  verifyToken,
  sideBarPermissionCheck("view", 268, "contractor_pre_site"),
  controllerWrapper(getContractorPreSite)
);

/**
 * @swagger
 * /api/contractorPreSite/get-contractor-pre-site/{id}:
 *   get:
 *     summary: Get a contractor pre-site by ID
 *     tags: [Contractor PreSite]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor pre-site
 *     responses:
 *       200:
 *         description: Contractor pre-site data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorPreSite'
 */

contractorPreSiteRoutes.get(
  "/get-contractor-pre-site/:id",
  verifyToken,
  sideBarPermissionCheck("view", 268, "contractor_pre_site"),
  controllerWrapper(getContractorPreSite)
);

/**
 * @swagger
 * /api/contractorPreSite/delete-contractor-pre-site/{id}:
 *   delete:
 *     summary: Delete a contractor pre-site by ID
 *     tags: [Contractor PreSite]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor pre-site to delete
 *     responses:
 *       200:
 *         description: Contractor pre-site was deleted
 */

contractorPreSiteRoutes.delete(
  "/delete-contractor-pre-site/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 268, "contractor_pre_site"),
  controllerWrapper(deleteContractorPreSite)
);

/**
 * @swagger
 * /api/contractorPreSite/send-pre-site:
 *   get:
 *     summary: Send a contractor pre-site
 *     tags: [Contractor PreSite]
 *     responses:
 *       200:
 *         description: The pre-site was successfully sent
 */

contractorPreSiteRoutes.post(
  "/send-pre-site",
  verifyToken,
  sideBarPermissionCheck("view", 268, "contractor_pre_site"),
  controllerWrapper(sendPreSite)
);

/**
 * @swagger
 * /api/contractorPreSite/fill-pre-site:
 *   post:
 *     summary: Fill out a contractor pre-site form
 *     tags: [Contractor PreSite]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorPreSite'
 *     responses:
 *       200:
 *         description: The form was successfully submitted
 */

contractorPreSiteRoutes.post("/fill-pre-site", controllerWrapper(fillForm));

export default contractorPreSiteRoutes;
