import { Router } from "express";
import {
  createUpdateCustomActionCreation,
  getCustomActionCreation,
  deleteCustomActionCreation,
  fetchLinkedTemplates,
  fetchGanttChart,
  getCustomActionCalender,
  getValidTasks,
  fetchEmployeeAccordingToGivenData,
  kanbanStatusUpdate,
  getKanbanData,
  getStructuredCustomActionCreation,
} from "../controller/customActionCreationController.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Custom Action Creation
 *   description: API operations for managing custom action creation
 */

const customActionCreationRoutes = Router();
/**
 * @swagger
 * /api/customActionCreation/create-update-custom-action-creation:
 *   post:
 *     tags: [Custom Action Creation]
 *     summary: Create or update a custom action creation
 *     description: Creates a new custom action creation or updates an existing one.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The ID of the custom action creation (for updates).
 *               template:
 *                 type: integer
 *                 description: The ID of the template.
 *               task_title:
 *                 type: string
 *                 description: The title of the task.
 *               tag:
 *                 type: integer
 *                 description: The ID of the tag.
 *               label:
 *                 type: integer
 *                 description: The ID of the label.
 *               task_description:
 *                 type: string
 *                 description: The description of the task.
 *               successor_task_template:
 *                 type: integer
 *                 description: The ID of the successor task template.
 *               predecessor_task_template:
 *                 type: integer
 *                 description: The ID of the predecessor task template.
 *               role:
 *                 type: integer
 *                 description: The ID of the role.
 *               skill:
 *                 type: integer
 *                 description: The ID of the skill.
 *               certificate:
 *                 type: integer
 *                 description: The ID of the certificate.
 *               responsible_person:
 *                 type: integer
 *                 description: The ID of the responsible person.
 *               collaborators:
 *                 type: string
 *                 description: The collaborators for the task.
 *               start_date:
 *                 type: string
 *                 format: date
 *                 description: The start date of the task.
 *               end_date:
 *                 type: string
 *                 format: date
 *                 description: The end date of the task.
 *               status:
 *                 type: string
 *                 description: The status of the task.
 *               priority_level:
 *                 type: string
 *                 description: The priority level of the task.
 *               attachment:
 *                 type: string
 *                 description: The attachment related to the task.
 *               task_budget:
 *                 type: integer
 *                 description: The budget for the task.
 *               task_resources:
 *                 type: string
 *                 description: The resources allocated to the task.
 *               organization:
 *                 type: integer
 *                 description: The ID of the organization.
 *               department:
 *                 type: integer
 *                 description: The ID of the department.
 *               deleted:
 *                 type: integer
 *                 description: Indicates if the record is deleted (0 or 1).
 *               created_by:
 *                 type: integer
 *                 description: The ID of the user who created the record.
 *               created_at:
 *                 type: string
 *                 format: date-time
 *                 description: The creation timestamp.
 *               updated_by:
 *                 type: integer
 *                 description: The ID of the user who last updated the record.
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *                 description: The last update timestamp.
 *             required:
 *               - template
 *               - task_title
 *               - responsible_person
 *               - collaborators
 *               - start_date
 *               - end_date
 *               - organization
 *           example:
 *             id: 1
 *             template: 1
 *             task_title: "Task Title"
 *             tag: 1
 *             label: 1
 *             task_description: "Description of the task."
 *             successor_task_template: 2
 *             predecessor_task_template: 3
 *             role: 1
 *             skill: 1
 *             certificate: 1
 *             responsible_person: 2
 *             collaborators: "Collaborator Names"
 *             start_date: "2024-08-01"
 *             end_date: "2024-08-15"
 *             status: "In Progress"
 *             priority_level: "High"
 *             attachment: "attachment_file.pdf"
 *             task_budget: 10000
 *             task_resources: "Resources"
 *             organization: 3
 *             department: 4
 *     responses:
 *       200:
 *         description: Custom Action Creation created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.post(
  "/create-update-custom-action-creation",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      235,
      "custom_action_creation"
    )(req, res, next);
  },
  controllerWrapper(createUpdateCustomActionCreation)
);

/**
 * @swagger
 * /api/customActionCreation/get-custom-action-creation:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get a list of custom action creation
 *     description: Retrieves a list of all custom action creation.
 *     responses:
 *       200:
 *         description: A list of custom action creation.
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.get(
  "/get-custom-action-creation",
  verifyToken,
  sideBarPermissionCheck("view", 235, "custom_action_creation"),
  controllerWrapper(getCustomActionCreation)
);

/**
 * @swagger
 * /api/customActionCreation/get-custom-action-creation/{id}:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get an custom action creation by ID
 *     description: Retrieves an custom action creation by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the custom action creation to retrieve.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: The custom action creation details.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Custom Action Creation not found.
 */

customActionCreationRoutes.get(
  "/get-custom-action-creation/:id",
  verifyToken,
  sideBarPermissionCheck("view", 235, "custom_action_creation"),
  controllerWrapper(getCustomActionCreation)
);

/**
 * @swagger
 * /api/customActionCreation/delete-custom-action-creation/{id}:
 *   delete:
 *     tags: [Custom Action Creation]
 *     summary: Delete an custom action creation by ID
 *     description: Deletes an custom action creation by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the custom action creation to delete.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Custom Action Creation deleted successfully.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Custom Action Creation not found.
 */

customActionCreationRoutes.delete(
  "/delete-custom-action-creation/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 235, "custom_action_creation"),
  controllerWrapper(deleteCustomActionCreation)
);

/**
 * @swagger
 * /api/customActionCreation/get-linked-templates:
 *   post:
 *     tags: [Custom Action Creation]
 *     summary: Get linked templated
 *     description: Retrieves an array of linked templates
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             organization : 1
 *             template : 1
 *     responses:
 *       200:
 *         description: A list of linked action templates
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.get(
  "/get-linked-templates",
  verifyToken,
  controllerWrapper(fetchLinkedTemplates)
);

/**
 * @swagger
 * /api/customActionCreation/get-gantt-chart:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get Gantt Chart
 *     description: Retrieves an array of linked templates
 *     responses:
 *       200:
 *         description: A list Gantt Chart
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.get(
  "/get-gantt-chart",
  verifyToken,
  controllerWrapper(fetchGanttChart)
);

/**
 * @swagger
 * /api/customActionCreation/get-action-calender:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get Task Calender
 *     description: Retrieves task timeline file for calender
 *     responses:
 *       200:
 *         description: path of action timeline file
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.get(
  "/get-action-calender",
  verifyToken,
  controllerWrapper(getCustomActionCalender)
);

/**
 * @swagger
 * /api/customActionCreation/get-linked-templates:
 *   post:
 *     tags: [Custom Action Creation]
 *     summary: Get linked templated
 *     description: Retrieves an array of linked templates
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             organization : 1
 *             template : 1
 *     responses:
 *       200:
 *         description: A list of linked action templates
 *       401:
 *         description: Unauthorized.
 */
customActionCreationRoutes.get(
  "/get-linked-templates",
  verifyToken,
  controllerWrapper(fetchLinkedTemplates)
);
/**
 * @swagger
 * /api/customActionCreation/get-valid-tasks:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get valid tasks
 *     description: Retrieves an array of valid tasks
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             task_id : 1
 *             organization : 1
 *             selected_type : "successor/predecessor"
 *     responses:
 *       200:
 *         description: A list of linked action templates
 *       401:
 *         description: Unauthorized.
 */
customActionCreationRoutes.get(
  "/get-valid-tasks",
  verifyToken,
  controllerWrapper(getValidTasks)
);
/**
 * @swagger
 * /api/customActionCreation/get-employee-with-data:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get valid tasks
 *     description: Retrieves an array of valid tasks
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             role : 1
 *             skill : 2
 *     responses:
 *       200:
 *         description:A list of employee according to
 *       401:
 *         description: Unauthorized.
 */
customActionCreationRoutes.get(
  "/get-employee-with-data",
  verifyToken,
  controllerWrapper(fetchEmployeeAccordingToGivenData)
);
/**
 * @swagger
 * /api/customActionCreation/update-kanban-status:
 *   post:
 *     tags: [Custom Action Creation]
 *     summary: Update status of tasks in kanban
 *     description: Update task status
 *     requestBody:
 *       required: true
 *       content:
 *         body:
 *           example:
 *             id : 1
 *             status : "To Do / In Progress / Done"
 *     responses:
 *       200:
 *         description:A list of employee according to
 *       401:
 *         description: Unauthorized.
 */
customActionCreationRoutes.post(
  "/update-kanban-status",
  verifyToken,
  controllerWrapper(kanbanStatusUpdate)
);

/**
 * @swagger
 * /api/customActionCreation/get-kanban-data:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get Kanban Data
 *     description: Retrieves the kanban data
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             organization : 1
 *     responses:
 *       200:
 *         description:Kanban data
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.get(
  "/get-kanban-data",
  verifyToken,
  controllerWrapper(getKanbanData)
);

/**
 * @swagger
 * /api/customActionCreation/get-structured-data:
 *   get:
 *     tags: [Custom Action Creation]
 *     summary: Get structured custom action data
 *     description: Retrieves the structured custom action data
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             organization : 1
 *     responses:
 *       200:
 *         description:structured custom action data
 *       401:
 *         description: Unauthorized.
 */

customActionCreationRoutes.get(
  "/get-structured-data",
  verifyToken,
  controllerWrapper(getStructuredCustomActionCreation)
);

export default customActionCreationRoutes;
