import { Router } from "express";
import {
  createUpdateCustomTaskCreation,
  getCustomTaskCreation,
  deleteCustomTaskCreation,
  fetchLinkedTemplates,
  fetchGanttChart,
  getCustomTaskCalender,
  getValidTasks,
  fetchEmployeeAccordingToGivenData,
  kanbanStatusUpdate,
  getKanbanData,
} from "../controller/customTaskCreationController.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Custom Task Creation
 *   description: API operations for managing custom task creation
 */

const customTaskCreationRoutes = Router();
/**
 * @swagger
 * /api/customTaskCreation/create-update-custom-task-creation:
 *   post:
 *     tags: [Custom Task Creation]
 *     summary: Create or update a custom task creation
 *     description: Creates a new custom task creation or updates an existing one.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The ID of the custom task creation (for updates).
 *               template:
 *                 type: integer
 *                 description: The ID of the template.
 *               task_title:
 *                 type: string
 *                 description: The title of the task.
 *               tag:
 *                 type: integer
 *                 description: The ID of the tag.
 *               label:
 *                 type: integer
 *                 description: The ID of the label.
 *               task_description:
 *                 type: string
 *                 description: The description of the task.
 *               successor_task_template:
 *                 type: integer
 *                 description: The ID of the successor task template.
 *               predecessor_task_template:
 *                 type: integer
 *                 description: The ID of the predecessor task template.
 *               role:
 *                 type: integer
 *                 description: The ID of the role.
 *               skill:
 *                 type: integer
 *                 description: The ID of the skill.
 *               certificate:
 *                 type: integer
 *                 description: The ID of the certificate.
 *               responsible_person:
 *                 type: integer
 *                 description: The ID of the responsible person.
 *               collaborators:
 *                 type: string
 *                 description: The collaborators for the task.
 *               start_date:
 *                 type: string
 *                 format: date
 *                 description: The start date of the task.
 *               end_date:
 *                 type: string
 *                 format: date
 *                 description: The end date of the task.
 *               status:
 *                 type: string
 *                 description: The status of the task.
 *               priority_level:
 *                 type: string
 *                 description: The priority level of the task.
 *               attachment:
 *                 type: string
 *                 description: The attachment related to the task.
 *               task_budget:
 *                 type: integer
 *                 description: The budget for the task.
 *               task_resources:
 *                 type: string
 *                 description: The resources allocated to the task.
 *               organization:
 *                 type: integer
 *                 description: The ID of the organization.
 *               department:
 *                 type: integer
 *                 description: The ID of the department.
 *               deleted:
 *                 type: integer
 *                 description: Indicates if the record is deleted (0 or 1).
 *               created_by:
 *                 type: integer
 *                 description: The ID of the user who created the record.
 *               created_at:
 *                 type: string
 *                 format: date-time
 *                 description: The creation timestamp.
 *               updated_by:
 *                 type: integer
 *                 description: The ID of the user who last updated the record.
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *                 description: The last update timestamp.
 *             required:
 *               - template
 *               - task_title
 *               - responsible_person
 *               - collaborators
 *               - start_date
 *               - end_date
 *               - organization
 *           example:
 *             id: 1
 *             template: 1
 *             task_title: "Task Title"
 *             tag: 1
 *             label: 1
 *             task_description: "Description of the task."
 *             successor_task_template: 2
 *             predecessor_task_template: 3
 *             role: 1
 *             skill: 1
 *             certificate: 1
 *             responsible_person: 2
 *             collaborators: "Collaborator Names"
 *             start_date: "2024-08-01"
 *             end_date: "2024-08-15"
 *             status: "In Progress"
 *             priority_level: "High"
 *             attachment: "attachment_file.pdf"
 *             task_budget: 10000
 *             task_resources: "Resources"
 *             organization: 3
 *             department: 4
 *     responses:
 *       200:
 *         description: Custom Task Creation created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

customTaskCreationRoutes.post(
  "/create-update-custom-task-creation",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      333,
      "custom_task_creation"
    )(req, res, next);
  },
  controllerWrapper(createUpdateCustomTaskCreation)
);

/**
 * @swagger
 * /api/customTaskCreation/get-custom-task-creation:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get a list of custom task creation
 *     description: Retrieves a list of all custom task creation.
 *     responses:
 *       200:
 *         description: A list of custom task creation.
 *       401:
 *         description: Unauthorized.
 */

customTaskCreationRoutes.get(
  "/get-custom-task-creation",
  verifyToken,
  sideBarPermissionCheck("view", 333, "custom_task_creation"),
  controllerWrapper(getCustomTaskCreation)
);

/**
 * @swagger
 * /api/customTaskCreation/get-custom-task-creation/{id}:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get an custom task creation by ID
 *     description: Retrieves an custom task creation by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the custom task creation to retrieve.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: The custom task creation details.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Custom Task Creation not found.
 */

customTaskCreationRoutes.get(
  "/get-custom-task-creation/:id",
  verifyToken,
  sideBarPermissionCheck("view", 333, "custom_task_creation"),
  controllerWrapper(getCustomTaskCreation)
);

/**
 * @swagger
 * /api/customTaskCreation/delete-custom-task-creation/{id}:
 *   delete:
 *     tags: [Custom Task Creation]
 *     summary: Delete an custom task creation by ID
 *     description: Deletes an custom task creation by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the custom task creation to delete.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Custom Task Creation deleted successfully.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Custom Task Creation not found.
 */

customTaskCreationRoutes.delete(
  "/delete-custom-task-creation/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 333, "custom_task_creation"),
  controllerWrapper(deleteCustomTaskCreation)
);

/**
 * @swagger
 * /api/customTaskCreation/get-linked-templates:
 *   post:
 *     tags: [Custom Task Creation]
 *     summary: Get linked templated
 *     description: Retrieves an array of linked templates
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             organization : 1
 *             template : 1
 *     responses:
 *       200:
 *         description: A list of linked task templates
 *       401:
 *         description: Unauthorized.
 */

customTaskCreationRoutes.get(
  "/get-linked-templates",
  verifyToken,
  controllerWrapper(fetchLinkedTemplates)
);

/**
 * @swagger
 * /api/customTaskCreation/get-gantt-chart:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get Gantt Chart
 *     description: Retrieves an array of linked templates
 *     responses:
 *       200:
 *         description: A list Gantt Chart
 *       401:
 *         description: Unauthorized.
 */

customTaskCreationRoutes.get(
  "/get-gantt-chart",
  verifyToken,
  controllerWrapper(fetchGanttChart)
);

/**
 * @swagger
 * /api/customTaskCreation/get-task-calender:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get Task Calender
 *     description: Retrieves task timeline file for calender
 *     responses:
 *       200:
 *         description: path of task timeline file
 *       401:
 *         description: Unauthorized.
 */

customTaskCreationRoutes.get(
  "/get-task-calender",
  verifyToken,
  controllerWrapper(getCustomTaskCalender)
);

/**
 * @swagger
 * /api/customTaskCreation/get-linked-templates:
 *   post:
 *     tags: [Custom Task Creation]
 *     summary: Get linked templated
 *     description: Retrieves an array of linked templates
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             organization : 1
 *             template : 1
 *     responses:
 *       200:
 *         description: A list of linked task templates
 *       401:
 *         description: Unauthorized.
 */
customTaskCreationRoutes.get(
  "/get-linked-templates",
  verifyToken,
  controllerWrapper(fetchLinkedTemplates)
);
/**
 * @swagger
 * /api/customTaskCreation/get-valid-tasks:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get valid tasks
 *     description: Retrieves an array of valid tasks
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             task_id : 1
 *             organization : 1
 *             selected_type : "successor/predecessor"
 *     responses:
 *       200:
 *         description: A list of linked task templates
 *       401:
 *         description: Unauthorized.
 */
customTaskCreationRoutes.get(
  "/get-valid-tasks",
  verifyToken,
  controllerWrapper(getValidTasks)
);
/**
 * @swagger
 * /api/customTaskCreation/get-employee-with-data:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get valid tasks
 *     description: Retrieves an array of valid tasks
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             role : 1
 *             skill : 2
 *     responses:
 *       200:
 *         description:A list of employee according to
 *       401:
 *         description: Unauthorized.
 */
customTaskCreationRoutes.get(
  "/get-employee-with-data",
  verifyToken,
  controllerWrapper(fetchEmployeeAccordingToGivenData)
);
/**
 * @swagger
 * /api/customTaskCreation/update-kanban-status:
 *   post:
 *     tags: [Custom Task Creation]
 *     summary: Update status of tasks in kanban
 *     description: Update task status
 *     requestBody:
 *       required: true
 *       content:
 *         body:
 *           example:
 *             id : 1
 *             status : "To Do / In Progress / Done"
 *     responses:
 *       200:
 *         description:A list of employee according to
 *       401:
 *         description: Unauthorized.
 */
customTaskCreationRoutes.post(
  "/update-kanban-status",
  verifyToken,
  controllerWrapper(kanbanStatusUpdate)
);

/**
 * @swagger
 * /api/customTaskCreation/get-kanban-data:
 *   get:
 *     tags: [Custom Task Creation]
 *     summary: Get Kanban Data
 *     description: Retrieves the kanban data
 *     requestBody:
 *       required: true
 *       content:
 *         query:
 *           example:
 *             organization : 1
 *     responses:
 *       200:
 *         description:Kanban data
 *       401:
 *         description: Unauthorized.
 */

customTaskCreationRoutes.get(
  "/get-kanban-data",
  verifyToken,
  controllerWrapper(getKanbanData)
);

export default customTaskCreationRoutes;
